/*
 * almanac.c -- GPS protocol almanac handling.
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/almanac.c,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 23:33 Apr 24 2000 kivinen
 *	  Last Modification : 12:21 Aug 19 2003 kivinen
 *	  Last check in     : $Date: 2003/08/18 20:07:01 $
 *	  Revision number   : $Revision: 1.8 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.65
 *	  Edit time	    : 25 min
 *
 *	  Description       : GPS protocol almanac handling
 *
 *	  $Log: almanac.c,v $
 *	  Revision 1.8  2003/08/18 20:07:01  kivinen
 *	  	Added support for GPS V. Fixed packet size and actual size
 *	  	comparisions.
 *
 *	  Revision 1.7  2001/08/23 15:43:05  kivinen
 *	  	Added support for eTrex format using format number 509 (my own
 *	  	number).
 *
 *	  Revision 1.6  2000/08/05 00:43:39  kivinen
 *	  	Updated to new interface.
 *
 *	  Revision 1.5  2000/07/26 17:09:04  kivinen
 *	  	Added almanac upload support.
 *
 *	  Revision 1.4  2000/07/21 22:16:45  kivinen
 *	  	Moved stuff to data.c.
 *
 *	  Revision 1.3  2000/07/06 23:05:36  kivinen
 *	  	Added initial upload support.
 *
 *	  Revision 1.2  2000/04/30 01:03:27  kivinen
 *	  	Updated to rev 03 document.
 *
 *	  Revision 1.1  2000/04/29 16:39:30  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#include "tgps.h"
#include "packet.h"
#include "data.h"

int tgps_almanac_data_in_d500(Tgps conn, TgpsPacket packet, TgpsAlmanac a);
int tgps_almanac_data_in_d501(Tgps conn, TgpsPacket packet, TgpsAlmanac a);
int tgps_almanac_data_in_d509(Tgps conn, TgpsPacket packet, TgpsAlmanac a);
int tgps_almanac_data_in_d550(Tgps conn, TgpsPacket packet, TgpsAlmanac a);
int tgps_almanac_data_in_d551(Tgps conn, TgpsPacket packet, TgpsAlmanac a);

TgpsPacket tgps_almanac_data_out_d500(Tgps conn, TgpsAlmanac a);
TgpsPacket tgps_almanac_data_out_d501(Tgps conn, TgpsAlmanac a);
TgpsPacket tgps_almanac_data_out_d509(Tgps conn, TgpsAlmanac a);
TgpsPacket tgps_almanac_data_out_d550(Tgps conn, TgpsAlmanac a);
TgpsPacket tgps_almanac_data_out_d551(Tgps conn, TgpsAlmanac a);

/* Process version 500 almanac packet */
int tgps_almanac_data_in_d500(Tgps conn, TgpsPacket packet, TgpsAlmanac a)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_INT, &a->wn,
		     TGPS_FORMAT_FLOAT, &a->toa,
		     TGPS_FORMAT_FLOAT, &a->af0,
		     TGPS_FORMAT_FLOAT, &a->af1,
		     TGPS_FORMAT_FLOAT, &a->e,
		     TGPS_FORMAT_FLOAT, &a->sqrta,
		     TGPS_FORMAT_FLOAT, &a->m0,
		     TGPS_FORMAT_FLOAT, &a->w,
		     TGPS_FORMAT_FLOAT, &a->omg0,
		     TGPS_FORMAT_FLOAT, &a->odot,
		     TGPS_FORMAT_FLOAT, &a->i,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 501 almanac packet */
int tgps_almanac_data_in_d501(Tgps conn, TgpsPacket packet, TgpsAlmanac a)
{
  size_t size;

  /* XXX Garmin GPS V, seems to be claiming to send 501 format, but actually
     uses 509 format. */
  if (packet->data_len == 48)
    return tgps_almanac_data_in_d509(conn, packet, a);
  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_INT, &a->wn,
		     TGPS_FORMAT_FLOAT, &a->toa,
		     TGPS_FORMAT_FLOAT, &a->af0,
		     TGPS_FORMAT_FLOAT, &a->af1,
		     TGPS_FORMAT_FLOAT, &a->e,
		     TGPS_FORMAT_FLOAT, &a->sqrta,
		     TGPS_FORMAT_FLOAT, &a->m0,
		     TGPS_FORMAT_FLOAT, &a->w,
		     TGPS_FORMAT_FLOAT, &a->omg0,
		     TGPS_FORMAT_FLOAT, &a->odot,
		     TGPS_FORMAT_FLOAT, &a->i,
		     TGPS_FORMAT_BYTE, &a->hlth,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 509 almanac packet XXX */
int tgps_almanac_data_in_d509(Tgps conn, TgpsPacket packet, TgpsAlmanac a)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_INT, &a->wn,
		     TGPS_FORMAT_FLOAT, &a->toa,
		     TGPS_FORMAT_FLOAT, &a->af0,
		     TGPS_FORMAT_FLOAT, &a->af1,
		     TGPS_FORMAT_FLOAT, &a->e,
		     TGPS_FORMAT_FLOAT, &a->sqrta,
		     TGPS_FORMAT_FLOAT, &a->m0,
		     TGPS_FORMAT_FLOAT, &a->w,
		     TGPS_FORMAT_FLOAT, &a->omg0,
		     TGPS_FORMAT_FLOAT, &a->odot,
		     TGPS_FORMAT_FLOAT, &a->i,
		     TGPS_FORMAT_BYTE, &a->hlth,
		     TGPS_FORMAT_BYTE_ARRAY, a->extra, 3,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 550 almanac packet */
int tgps_almanac_data_in_d550(Tgps conn, TgpsPacket packet, TgpsAlmanac a)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR, &a->svid,
		     TGPS_FORMAT_INT, &a->wn,
		     TGPS_FORMAT_FLOAT, &a->toa,
		     TGPS_FORMAT_FLOAT, &a->af0,
		     TGPS_FORMAT_FLOAT, &a->af1,
		     TGPS_FORMAT_FLOAT, &a->e,
		     TGPS_FORMAT_FLOAT, &a->sqrta,
		     TGPS_FORMAT_FLOAT, &a->m0,
		     TGPS_FORMAT_FLOAT, &a->w,
		     TGPS_FORMAT_FLOAT, &a->omg0,
		     TGPS_FORMAT_FLOAT, &a->odot,
		     TGPS_FORMAT_FLOAT, &a->i,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 551 almanac packet */
int tgps_almanac_data_in_d551(Tgps conn, TgpsPacket packet, TgpsAlmanac a)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR, &a->svid,
		     TGPS_FORMAT_INT, &a->wn,
		     TGPS_FORMAT_FLOAT, &a->toa,
		     TGPS_FORMAT_FLOAT, &a->af0,
		     TGPS_FORMAT_FLOAT, &a->af1,
		     TGPS_FORMAT_FLOAT, &a->e,
		     TGPS_FORMAT_FLOAT, &a->sqrta,
		     TGPS_FORMAT_FLOAT, &a->m0,
		     TGPS_FORMAT_FLOAT, &a->w,
		     TGPS_FORMAT_FLOAT, &a->omg0,
		     TGPS_FORMAT_FLOAT, &a->odot,
		     TGPS_FORMAT_FLOAT, &a->i,
		     TGPS_FORMAT_BYTE, &a->hlth,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process almanac packet */
int tgps_almanac_data_in(Tgps conn, TgpsPacket packet, TgpsAlmanac a)
{
  int ret;

  tgps_clear_almanac(a);

  if (tgps_is_supported(conn, 'D', 500))
    ret = tgps_almanac_data_in_d500(conn, packet, a);
  else if (tgps_is_supported(conn, 'D', 501))
    ret = tgps_almanac_data_in_d501(conn, packet, a);
  else if (tgps_is_supported(conn, 'D', 509))
    ret = tgps_almanac_data_in_d509(conn, packet, a);
  else if (tgps_is_supported(conn, 'D', 550))
    ret = tgps_almanac_data_in_d550(conn, packet, a);
  else if (tgps_is_supported(conn, 'D', 551))
    ret = tgps_almanac_data_in_d551(conn, packet, a);
  else if (packet->data_len == 44)
    ret = tgps_almanac_data_in_d500(conn, packet, a);
  else if (packet->data_len == 45) /* version 550 has same length */
    ret = tgps_almanac_data_in_d501(conn, packet, a);
  else if (packet->data_len == 46)
    ret = tgps_almanac_data_in_d551(conn, packet, a);
  else if (packet->data_len == 48)
    ret = tgps_almanac_data_in_d509(conn, packet, a);
  else
    {
      fprintf(stderr, "Unknown almanac type\n");
      return 0;
    }
  if (ret != 0)
    conn->packet_count--;
  return ret;
}

/* Send version 500 waypoint data packet */
TgpsPacket tgps_almanac_data_out_d500(Tgps conn, TgpsAlmanac a)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_ALMANAC_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_INT, a->wn,
		     TGPS_FORMAT_FLOAT, a->toa,
		     TGPS_FORMAT_FLOAT, a->af0,
		     TGPS_FORMAT_FLOAT, a->af1,
		     TGPS_FORMAT_FLOAT, a->e,
		     TGPS_FORMAT_FLOAT, a->sqrta,
		     TGPS_FORMAT_FLOAT, a->m0,
		     TGPS_FORMAT_FLOAT, a->w,
		     TGPS_FORMAT_FLOAT, a->omg0,
		     TGPS_FORMAT_FLOAT, a->odot,
		     TGPS_FORMAT_FLOAT, a->i,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 501 waypoint data packet */
TgpsPacket tgps_almanac_data_out_d501(Tgps conn, TgpsAlmanac a)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_ALMANAC_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_INT, a->wn,
		     TGPS_FORMAT_FLOAT, a->toa,
		     TGPS_FORMAT_FLOAT, a->af0,
		     TGPS_FORMAT_FLOAT, a->af1,
		     TGPS_FORMAT_FLOAT, a->e,
		     TGPS_FORMAT_FLOAT, a->sqrta,
		     TGPS_FORMAT_FLOAT, a->m0,
		     TGPS_FORMAT_FLOAT, a->w,
		     TGPS_FORMAT_FLOAT, a->omg0,
		     TGPS_FORMAT_FLOAT, a->odot,
		     TGPS_FORMAT_FLOAT, a->i,
		     TGPS_FORMAT_BYTE, a->hlth,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 509 waypoint data packet XXX */
TgpsPacket tgps_almanac_data_out_d509(Tgps conn, TgpsAlmanac a)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_ALMANAC_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_INT, a->wn,
		     TGPS_FORMAT_FLOAT, a->toa,
		     TGPS_FORMAT_FLOAT, a->af0,
		     TGPS_FORMAT_FLOAT, a->af1,
		     TGPS_FORMAT_FLOAT, a->e,
		     TGPS_FORMAT_FLOAT, a->sqrta,
		     TGPS_FORMAT_FLOAT, a->m0,
		     TGPS_FORMAT_FLOAT, a->w,
		     TGPS_FORMAT_FLOAT, a->omg0,
		     TGPS_FORMAT_FLOAT, a->odot,
		     TGPS_FORMAT_FLOAT, a->i,
		     TGPS_FORMAT_BYTE, a->hlth,
		     TGPS_FORMAT_BYTE_ARRAY, a->extra, 3,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 550 waypoint data packet */
TgpsPacket tgps_almanac_data_out_d550(Tgps conn, TgpsAlmanac a)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_ALMANAC_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR, a->svid,
		     TGPS_FORMAT_INT, a->wn,
		     TGPS_FORMAT_FLOAT, a->toa,
		     TGPS_FORMAT_FLOAT, a->af0,
		     TGPS_FORMAT_FLOAT, a->af1,
		     TGPS_FORMAT_FLOAT, a->e,
		     TGPS_FORMAT_FLOAT, a->sqrta,
		     TGPS_FORMAT_FLOAT, a->m0,
		     TGPS_FORMAT_FLOAT, a->w,
		     TGPS_FORMAT_FLOAT, a->omg0,
		     TGPS_FORMAT_FLOAT, a->odot,
		     TGPS_FORMAT_FLOAT, a->i,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 551 waypoint data packet */
TgpsPacket tgps_almanac_data_out_d551(Tgps conn, TgpsAlmanac a)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_ALMANAC_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR, a->svid,
		     TGPS_FORMAT_INT, a->wn,
		     TGPS_FORMAT_FLOAT, a->toa,
		     TGPS_FORMAT_FLOAT, a->af0,
		     TGPS_FORMAT_FLOAT, a->af1,
		     TGPS_FORMAT_FLOAT, a->e,
		     TGPS_FORMAT_FLOAT, a->sqrta,
		     TGPS_FORMAT_FLOAT, a->m0,
		     TGPS_FORMAT_FLOAT, a->w,
		     TGPS_FORMAT_FLOAT, a->omg0,
		     TGPS_FORMAT_FLOAT, a->odot,
		     TGPS_FORMAT_FLOAT, a->i,
		     TGPS_FORMAT_BYTE, a->hlth,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send almanac data packet */
TgpsPacket tgps_almanac_data_out(Tgps conn, TgpsAlmanac a)
{
  if (tgps_is_supported(conn, 'D', 500))
    return tgps_almanac_data_out_d500(conn, a);
  else if (tgps_is_supported(conn, 'D', 501))
    return tgps_almanac_data_out_d501(conn, a);
  else if (tgps_is_supported(conn, 'D', 509))
    return tgps_almanac_data_out_d509(conn, a);
  else if (tgps_is_supported(conn, 'D', 550))
    return tgps_almanac_data_out_d550(conn, a);
  else if (tgps_is_supported(conn, 'D', 551))
    return tgps_almanac_data_out_d551(conn, a);
  else
    {
      fprintf(stderr, "Unknown almanac protocol format\n");
      return NULL;
    }
}
