/*
 * input.c -- GPS protocol input code.
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/input.c,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 00:21 May 22 2000 kivinen
 *	  Last Modification : 18:33 Feb  5 2004 kivinen
 *	  Last check in     : $Date: 2003/08/18 20:10:34 $
 *	  Revision number   : $Revision: 1.11 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.752
 *	  Edit time	    : 307 min
 *
 *	  Description       : GPS protocol input code
 *
 *	  $Log: input.c,v $
 *	  Revision 1.11  2003/08/18 20:10:34  kivinen
 *	  	Added support for GPS V. Fixed packet size and actual size
 *	  	comparisions.
 *
 *	  Revision 1.10  2001/08/23 15:55:57  kivinen
 *	  	Added code to read extra data from the almanac and track
 *	  	records. Fixed subclass handing to read hexbytes.
 *
 *	  Revision 1.9  2000/10/02 15:43:28  kivinen
 *	  	Changed to use tgps_print.
 *
 *	  Revision 1.8  2000/08/05 00:45:04  kivinen
 *	  	Removed warnings.
 *
 *	  Revision 1.7  2000/07/26 17:34:39  kivinen
 *	  	Fixed clock info reading.
 *
 *	  Revision 1.6  2000/07/26 17:09:41  kivinen
 *	  	Added almanac input support.
 *
 *	  Revision 1.5  2000/07/23 20:23:52  kivinen
 *	  	Added tgps_get_pos_double. Added pvt format.
 *
 *	  Revision 1.4  2000/07/21 22:16:13  kivinen
 *	  	Moved stuff to tgps.c.
 *
 *	  Revision 1.3  2000/07/17 18:09:20  kivinen
 *	  	Added new format for newtrack.
 *
 *	  Revision 1.2  2000/07/12 22:20:06  kivinen
 *	  	Final upload support.
 *
 *	  Revision 1.1  2000/07/06 23:06:51  kivinen
 *	  	Added initial upload support.
 *
 *	  $EndLog$
 */

#include "tgps.h"
#include "enum.h"
#include "input.h"

typedef int (*TgpsLineFunc)(Tgps conn, void *data, char *ptr);
typedef int (*TgpsOptionFunc)(Tgps conn, void *data, char *ptr, void *data_ptr,
			      size_t data_len);

/* Return 0 if error, and negative number if input line ended before all
   functions ware used. Return 1 if ok, return 2 if there is extra data after
   the item. */
int tgps_get_and_parse_line(Tgps conn, void *data, ...)
{
  char buffer[1024];
  char *item, *p;
  size_t len;
  int ret;
  TgpsLineFunc fptr;
  va_list ap;
  int items;
  
  va_start(ap, data);

  if (conn->input_buffer_len == -1)
    {
      if (tgps_gets(conn))
	{
	  va_end(ap);
	  conn->input_buffer_len = 0;
	}
      else
	{
	  conn->input_buffer_len = strlen(conn->input_buffer);
	  len = conn->input_buffer_len;
	  while (conn->input_buffer[len - 1] == '\n' ||
		 conn->input_buffer[len - 1] == '\r')
	    {
	      conn->input_buffer[len - 1] = '\0';
	      len--;
	    }
	}
    }
  if (conn->input_buffer_len == 0)
    return 0;

  memcpy(buffer, conn->input_buffer, conn->input_buffer_len + 1);
  p = buffer;
  items = 1;
  while ((item = strsep(&p, "\t")) != NULL)
    {
      while (isspace(*item))
	item++;
      len = strlen(item);
      while (isspace(item[len - 1]))
	{
	  len--;
	}
      item[len] = '\0';
      fptr = va_arg(ap, TgpsLineFunc);
      if (fptr == NULL)
	{
	  va_end(ap);
	  conn->input_buffer_len = -1;
	  return 2;
	}
      ret = (*fptr)(conn, data, item);
      if (ret == 0)
	{
	  va_end(ap);
	  return 0;
	}
      items++;
    }
  fptr = va_arg(ap, TgpsLineFunc);
  if (fptr == NULL)
    {
      va_end(ap);
      conn->input_buffer_len = -1;
      return 1;
    }
  va_end(ap);
  return -items;
}

/* Return 0 in case of error, and 1 if ok.
   ... = is list of options, which are
   text, function, pointer_where_to_store, size. */
int tgps_get_and_parse_options(Tgps conn, void *data, char *ptr, ...)
{
  TgpsOptionFunc fptr;
  size_t data_size, len;
  const char *name;
  void *data_ptr;
  va_list ap;
  char *item;
  int ret;

  while ((item = strsep(&ptr, ",")) != NULL)
    {
      while (isspace(*item))
	item++;
      len = strlen(item);
      while (isspace(item[len - 1]))
	{
	  len--;
	}
      item[len] = '\0';
      va_start(ap, ptr);
      while (1)
	{
	  name = va_arg(ap, const char *);
	  if (name == NULL)	/* Ignore unknown option */
	    break;
	  fptr = va_arg(ap, TgpsOptionFunc);
	  data_ptr = va_arg(ap, void *);
	  data_size = va_arg(ap, int);
	  len = strlen(name);
	  if (strncasecmp(name, item, len) == 0)
	    {
	      ret = (*fptr)(conn, data, item + len, data_ptr, data_size);
	      if (ret == 0)
		{
		  va_end(ap);
		  return 0;
		}
	      break;
	    }
	}
      va_end(ap);
    }
  return 1;
}

/* Get position double. Return 0 in case of error, and 1 if ok. */
int tgps_get_pos_double(Tgps conn, char *ptr, double *dret)
{
  int i1, i2;
  double d;

  if (sscanf(ptr, "%d%d'%lf\"", &i1, &i2, &d) == 3)
    {
      if (i1 > 180 || i1 < -180 || i2 >= 60 || i2 < 0 ||
	  d >= 60.0 || d < 0.0)
	return 0;
      if (i1 > 0)
	d = (double) i1 + ((double) i2 / 60.0) + (d / 3600.0);
      else
	d = (double) i1 - ((double) i2 / 60.0) - (d / 3600.0);
      *dret = d;
      return 1;
    }
  if (sscanf(ptr, "%d%lf'", &i1, &d) == 2)
    {
      if (i1 > 180 || i1 < -180 || d >= 60.0 || d < 0.0)
	return 0;
      if (i1 > 0)
	d = (double) i1 + (d / 60.0);
      else
	d = (double) i1 - (d / 60.0);
      *dret = d;
      return 1;
    }
  if (sscanf(ptr, "%lf", &d) == 1)
    {
      if (d > 180.0 || d < -180.0)
	return 0;
      *dret = d;
      return 1;
    }
  return 0;
}

/* Get position. Return 0 in case of error, and 1 if ok. */
int tgps_get_pos(Tgps conn, char *ptr, long *l)
{
  double d;
  int ret;

  ret = tgps_get_pos_double(conn, ptr, &d);
  if (ret)
    *l = tgps_deg_to_semicircle(d);
  return ret;
}

struct TgpsEnumTableRec tgps_month[] =
{
  { "jan", 1 },
  { "feb", 2 },
  { "mar", 3 },
  { "apr", 4 },
  { "may", 5 },
  { "jun", 6 },
  { "jul", 7 },
  { "aug", 8 },
  { "sep", 9 },
  { "oct", 10 },
  { "nov", 11 },
  { "dec", 12 }
};

int tgps_month_cnt = sizeof(tgps_month) / sizeof(tgps_month[0]);

/* Convert month name to number */
int tgps_month_to_number(const char *name)
{
  return tgps_string_to_number(tgps_month, tgps_month_cnt, name);
}

/* Get time from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_time(Tgps conn, void *data, char *ptr,
		  void *data_ptr, size_t data_len)
{
  int year, mon = -1, day, hour, min, sec;
  char mon_str[5], tz[5];
  struct tm tm;
  time_t t;

  mon_str[0] = '\0';
  tz[0] = '\0';
  if (sscanf(ptr, "%d-%d-%d %d:%d:%d %5s",
	     &year, &mon, &day, &hour, &min, &sec, tz) == 7)
    goto ok;
  if (sscanf(ptr, "%d/%d/%d %d:%d:%d",
	     &mon, &day, &year, &hour, &min, &sec) == 6)
    goto ok;
  if (sscanf(ptr, "%4d%2d%2d %2d%2d%2d", 
	     &year, &mon, &day, &hour, &min, &sec) == 6)
    goto ok;
  if (sscanf(ptr, "%4d%2d%2d%2d%2d%2d", 
	     &year, &mon, &day, &hour, &min, &sec) == 6)
    goto ok;
  if (sscanf(ptr, "%*s %4s %d %d:%d:%d %4s %d",
	     mon_str, &day, &hour, &min, &sec, tz, &year) == 7)
    goto ok;
  return 0;

 ok:
  if (tz[0])
    {
      /* XXX add timezone support */
      if (strcasecmp(tz, "GMT") != 0)
	{
	  fprintf(stderr, "Warning unknonw timezone in tgps_get_time: %s", tz);
	}
    }
  if (mon == -1)
    {
      mon = tgps_month_to_number(mon_str);
      if (mon == -1)
	return 0;
    }
  if (sec < 0 || sec > 59 || min < 0 || min > 59 || hour < 0 || hour > 23)
    return 0;
  if (year < 100)
    year += 1900;
  if (mon < 1 || mon > 12 || day < 1 || day > 31 || year < 1900)
    return 0;

  tm.tm_sec = sec;
  tm.tm_min = min;
  tm.tm_hour = hour;
  tm.tm_mday = day;
  tm.tm_mon = mon - 1;
  tm.tm_year = year - 1900;
  tm.tm_wday = 0;
  tm.tm_yday = 0;
  tm.tm_isdst = 0;
  tm.tm_gmtoff = 0;
  tm.tm_zone = tz;
  t = mktime(&tm);
  *((unsigned long *) data_ptr) = t - TGPS_BASE_TIME;
  return 1;
}

/* Get float from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_float(Tgps conn, void *data, char *ptr,
		   void *data_ptr, size_t data_len)
{
  char *endptr;
  double d;

  d = strtod(ptr, &endptr);

  /* Is it empty */
  if (ptr == endptr)
    return 0;
  while (isspace(*endptr))
    endptr++;
  /* Is there junk at the end */
  if (*endptr)
    return 0;

  *((float *) data_ptr) = (float) d;
  return 1;
}

/* Get 3 floats from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_float_3(Tgps conn, char *ptr, float *f1, float *f2, float *f3)
{
  char *endptr;
  double d;

  d = strtod(ptr, &endptr);

  /* Is it empty */
  if (ptr == endptr)
    return 0;
  while (isspace(*endptr))
    endptr++;
  /* Is there next item */
  if (*endptr != ',')
    return 0;
  *f1 = (float) d;

  ptr = endptr + 1;
  d = strtod(ptr, &endptr);

  /* Is it empty */
  if (ptr == endptr)
    return 0;
  while (isspace(*endptr))
    endptr++;
  /* Is there next item */
  if (*endptr != ',')
    return 0;
  *f2 = (float) d;

  ptr = endptr + 1;
  d = strtod(ptr, &endptr);

  /* Is it empty */
  if (ptr == endptr)
    return 0;
  while (isspace(*endptr))
    endptr++;
  /* Is there junk after the item */
  if (*endptr)
    return 0;
  *f3 = (float) d;
  return 1;
}

/* Get long integer from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_long(Tgps conn, void *data, char *ptr,
		  void *data_ptr, size_t data_len)
{
  char *endptr;
  long l;

  l = strtol(ptr, &endptr, 0);

  /* Is it empty */
  if (ptr == endptr)
    return 0;
  while (isspace(*endptr))
    endptr++;
  /* Is there junk at the end */
  if (*endptr)
    return 0;

  *((long *) data_ptr) = l;
  return 1;
}

/* Get int from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_word(Tgps conn, void *data, char *ptr,
		  void *data_ptr, size_t data_len)
{
  long l;

  if (!tgps_get_long(conn, data, ptr, &l, sizeof(l)))
    return 0;

  /* Is the value too large */
  if (l > 0xffff || l < 0)
    return 0;

  *((int *) data_ptr) = (l & 0xffff);
  return 1;
}

/* Get byte from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_byte(Tgps conn, void *data, char *ptr,
		  void *data_ptr, size_t data_len)
{
  long l;

  if (!tgps_get_long(conn, data, ptr, &l, sizeof(l)))
    return 0;

  /* Is the value too large */
  if (l > 255 || l < 0)
    return 0;

  *((char *) data_ptr) = (l & 0xff);
  return 1;
}

struct TgpsEnumTableRec tgps_garmin_boolean[] =
{
  { "false", 0 },
  { "off", 0 },
  { "true", 1 },
  { "on", 1 }
};

int tgps_garmin_boolean_cnt =
sizeof(tgps_garmin_boolean) / sizeof(tgps_garmin_boolean[0]);

/* Get boolean from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_boolean(Tgps conn, void *data, char *ptr,
		     void *data_ptr, size_t data_len)
{
  int i;

  i = tgps_string_to_number(tgps_garmin_boolean,
			    tgps_garmin_boolean_cnt,
			    ptr);

  /* Is the value invalid */
  if (i == -1)
    return 0;
  
  *((char *) data_ptr) = i;
  return 1;
}

/* Get bytes from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_bytes(Tgps conn, void *data, char *ptr,
		   void *data_ptr, size_t data_len)
{
  size_t len;

  len = strlen(ptr);
  if (len > data_len)
    len = data_len;
  /* Note, do not copy null at the end */
  memcpy(data_ptr, ptr, len);
  return 1;
}

/* Get hexbytes from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_hexbytes(Tgps conn, void *data, char *ptr,
		      void *data_ptr, size_t data_len)
{
  unsigned char *p;
  int i;

  if (strncmp("0x", ptr, 2) == 0)
    ptr += 2;
  memset(data_ptr, 0, data_len);
  p = data_ptr;
  for(i = 0; i < data_len; i++)
    {
      if (!*ptr)
	break;
      if (!isxdigit(ptr[0]) || !isxdigit(ptr[1]))
	return 0;
      *p++ =
	(isdigit(ptr[0]) ? (ptr[0] - '0') : (tolower(ptr[0]) - 'a' + 10)) <<4 |
	(isdigit(ptr[1]) ? (ptr[1] - '0') : (tolower(ptr[1]) - 'a' + 10));
      ptr += 2;
    }
  return 1;
}

/* Get 3 hex bytes from the buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_hexbytes_3(Tgps conn, void *data, char *ptr,
		      void *data_ptr, size_t data_len)
{
  return tgps_get_hexbytes(conn, data, ptr, data_ptr, 3);
}

/* Get string from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_string(Tgps conn, void *data, char *ptr,
		    void *data_ptr, size_t data_len)
{
  *((char **) data_ptr) = strdup(ptr);
  return 1;
}

/* Get color from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_color(Tgps conn, void *data, char *ptr,
		       void *data_ptr, size_t data_len)
{
  int c;

  c = tgps_string_to_color(ptr);
  if (c <= 0)
    return 0;

  if (tgps_is_supported(conn, 'D', 107))
    {
      if (c == 17 || c == 25)
	c = 1;
      else if (c == 18 || c == 26)
	c = 2;
      else if (c == 20 || c == 28)
	c = 3;
      else
	c = 0;
    }
  else
    {
      if (c >= 16)
	c -= 16;
    }
  *((char *) data_ptr) = c;
  return 1;
}

/* Check for w1 format. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_check_for_w1(Tgps conn, TgpsWaypoint w, char *ptr)
{
  if (strcasecmp(ptr, "W1") == 0)
    return 1;
  return 0;
}

/* Get ident string. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_ident(Tgps conn, TgpsWaypoint w, char *ptr)
{
  size_t len;

  len = strlen(ptr);
  if (len > 0)
    w->wpt_ident = strdup(ptr);
  strncpy(w->ident, ptr, sizeof(w->ident) - 1);
  return 1;
}

/* Get link or comment field. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_lnk_or_comment(Tgps conn, TgpsWaypoint w, char *ptr)
{
  if (strncmp(ptr, "-> ", 3) == 0)
    {
      w->lnk_ident = strdup(ptr + 3);
    }
  else
    {
      strncpy(w->comment, ptr, sizeof(w->comment) - 1);
      w->comment_var = strdup(ptr);
    }
  return 1;
}

/* Get latitude. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_lat(Tgps conn, TgpsWaypoint w, char *ptr)
{
  return tgps_get_pos(conn, ptr, &w->lat);
}

/* Get longitude. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_lon(Tgps conn, TgpsWaypoint w, char *ptr)
{
  return tgps_get_pos(conn, ptr, &w->lon);
}

/* Get symbol id from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_symbol(Tgps conn, void *data, char *ptr,
			void *data_ptr, size_t data_len)
{
  TgpsWaypoint w = data;
  int s;

  s = tgps_string_to_symbol(TGPS_GARMIN_SYMBOLS_FULL, ptr);
  if (s == -1)
    return 0;
  
  if (tgps_is_supported(conn, 'D', 103) ||
      tgps_is_supported(conn, 'D', 403) ||
      tgps_is_supported(conn, 'D', 107))
    w->smbl_type = TGPS_GARMIN_SYMBOLS_16;
  else
    w->smbl_type = TGPS_GARMIN_SYMBOLS_FULL;
  if (w->smbl_type == TGPS_GARMIN_SYMBOLS_16)
    s = tgps_string_to_symbol(w->smbl_type, ptr);

  if (s == -1)
    s = 0;

  w->smbl16 = s;
  w->smbl = s;
  return 1;
}

/* Get display type from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_display(Tgps conn, void *data, char *ptr,
			 void *data_ptr, size_t data_len)
{
  TgpsWaypoint w = data;

  if (strcasecmp(ptr, "S+N") == 0)
    w->display = 0;
  else if (strcasecmp(ptr, "S") == 0)
    w->display = 1;
  else if (strcasecmp(ptr, "S+C") == 0)
    w->display = 2;

  if (tgps_is_supported(conn, 'D', 104) ||
      tgps_is_supported(conn, 'D', 155))
    {
      if (w->display == 2)
	w->display = 5;
      if (w->display == 0)
	w->display = 3;
    }
  return 1;
}

/* Get city from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_city(Tgps conn, void *data, char *ptr,
		      void *data_ptr, size_t data_len)
{
  TgpsWaypoint w = data;
  int len;

  len = strlen(ptr);
  if (len > sizeof(w->city) - 1)
    len = sizeof(w->city) - 1;

  strncpy(w->city, ptr, len);
  w->city_var = strdup(ptr);
  return 1;
}

/* Get alt from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_alt(Tgps conn, void *data, char *ptr,
		     void *data_ptr, size_t data_len)
{
  TgpsWaypoint w = data;
  float f;

  if (!tgps_get_float(conn, data, ptr, &f, sizeof(f)))
    return 0;

  w->alt = (int) f;
  w->altf = f;
  return 1;
}

/* Get options. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_options(Tgps conn, TgpsWaypoint w, char *ptr)
{
  return
    tgps_get_and_parse_options(conn, w, ptr,
			       "smbl=", tgps_wpt_get_symbol, NULL, 0, 
			       "dspl=", tgps_wpt_get_display, NULL, 0, 
			       "color=", tgps_get_color, &w->color, 0,
			       "attr=", tgps_get_byte, &w->attribute, 0, 
			       "prxdst=", tgps_get_float, &w->proximity, 0,
			       "class=", tgps_get_byte, &w->class, 0, 
			       "subclass=", tgps_get_hexbytes,
			       w->subclass, sizeof(w->subclass),
			       "cc=", tgps_get_bytes,
			       w->cc, sizeof(w->cc) - 1, 
			       "facility=", tgps_get_string,
			       &w->facility_var, 0,
			       "city=", tgps_wpt_get_city, NULL, 0,
			       "state=", tgps_get_bytes,
			       w->state, sizeof(w->state) - 1,
			       "name=", tgps_get_bytes,
			       w->name, sizeof(w->name) - 1,
			       "address=", tgps_get_string, &w->addr_var, 0, 
			       "crossroad=", tgps_get_string,
			       &w->cross_road_var, 0, 
			       "prxidx=", tgps_get_word, &w->idx, 0,
			       "alt=", tgps_wpt_get_alt, NULL, 0,
			       "depth=", tgps_get_float, &w->dpth, 0,
			       "time=", tgps_get_time,
			       &w->unused_creation_time, 0,
			       "ete=", tgps_get_long, &w->ete, 0,
			       NULL);
}

/* Get waypoint structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_waypoint_v1(Tgps conn, TgpsWaypoint w)
{
  int ret;
  ret = tgps_get_and_parse_line(conn, w,
				tgps_wpt_check_for_w1,
				tgps_wpt_get_ident,
				tgps_wpt_get_lnk_or_comment,
				tgps_wpt_get_lat, 
				tgps_wpt_get_lon,
				tgps_wpt_get_options,
				NULL);
  /* Return ok, even if options field is missing */
  if (ret == -6)
    {
      conn->input_buffer_len = -1;
      return 1;
    }
  return ret;
}

/* Check for w format. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_check_for_w(Tgps conn, TgpsWaypoint w, char *ptr)
{
  if (strcasecmp(ptr, "W") == 0)
    return 1;
  return 0;
}

/* Get time from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_time(Tgps conn, void *data, char *ptr,
		     void *data_ptr, size_t data_len)
{
  TgpsWaypoint w = data;

  return tgps_get_time(conn, data, ptr, &w->unused_creation_time,
		       sizeof(w->unused_creation_time));
}

/* Get waypoint structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_waypoint_gpstrans(Tgps conn, TgpsWaypoint w)
{
  return tgps_get_and_parse_line(conn, w,
				 tgps_wpt_check_for_w,
				 tgps_wpt_get_ident,
				 tgps_wpt_get_lnk_or_comment,
				 tgps_wpt_get_time,
				 tgps_wpt_get_lat, 
				 tgps_wpt_get_lon,
				 NULL);
}

/* Get waypoint structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_waypoint(Tgps conn, TgpsWaypoint w)
{
  int ret;

  ret = tgps_get_waypoint_v1(conn, w);
  if (ret)
    return ret;
  return tgps_get_waypoint_gpstrans(conn, w);
}

/* Check for r format. Return 0 in case of error, and 1 if ok. */
int tgps_rhdr_check_for_r(Tgps conn, TgpsRouteHeader r, char *ptr)
{
  if (strcasecmp(ptr, "R") == 0)
    return 1;
  return 0;
}

/* Get route number. Return 0 in case of error, and 1 if ok. */
int tgps_rhdr_get_route_number(Tgps conn, TgpsRouteHeader r, char *ptr)
{
  return tgps_get_byte(conn, r, ptr, &r->number, sizeof(r->number));
}

/* Get route header comment field. Return 0 in case of error, and 1 if ok. */
int tgps_rhdr_get_route_comment(Tgps conn, TgpsRouteHeader r, char *ptr)
{
  strncpy(r->comment, ptr, sizeof(r->comment) - 1);
  r->rte_ident = strdup(ptr);
  return 1;
}

/* Get route header structure in gpstrans format. Return 0 in case of error,
   and 1 if ok. */
int tgps_get_route_header_gpstrans(Tgps conn, TgpsRouteHeader r)
{
  return tgps_get_and_parse_line(conn, r,
				 tgps_rhdr_check_for_r,
				 tgps_rhdr_get_route_number,
				 tgps_rhdr_get_route_comment,
				 NULL);
}

/* Check for r1 format. Return 0 in case of error, and 1 if ok. */
int tgps_rhdr_check_for_r1(Tgps conn, TgpsRouteHeader r, char *ptr)
{
  if (strcasecmp(ptr, "R1") == 0)
    return 1;
  return 0;
}

/* Get route header structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_route_header_v1(Tgps conn, TgpsRouteHeader r)
{
  return tgps_get_and_parse_line(conn, r,
				 tgps_rhdr_check_for_r1,
				 tgps_rhdr_get_route_number,
				 tgps_rhdr_get_route_comment,
				 NULL);
}

/* Get route header structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_route_header(Tgps conn, TgpsRouteHeader r)
{
  int ret;
  ret = tgps_get_route_header_v1(conn, r);
  if (ret)
    return ret;
  return tgps_get_route_header_gpstrans(conn, r);
}

/* Check for k1 format. Return 0 in case of error, and 1 if ok. */
int tgps_rlnk_check_for_k1(Tgps conn, TgpsRouteLnk r, char *ptr)
{
  if (strcasecmp(ptr, "K1") == 0)
    return 1;
  return 0;
}

/* Get route link ident string. Return 0 in case of error, and 1 if ok. */
int tgps_rlnk_get_ident(Tgps conn, TgpsRouteLnk r, char *ptr)
{
  r->ident = strdup(ptr);
  return 1;
}

/* Get options. Return 0 in case of error, and 1 if ok. */
int tgps_rlnk_get_options(Tgps conn, TgpsRouteLnk r, char *ptr)
{
  return
    tgps_get_and_parse_options(conn, r, ptr,
			       "class=", tgps_get_byte, &r->class, 0, 
			       "subclass=", tgps_get_hexbytes, r->subclass,
			       sizeof(r->subclass),
			       NULL);
}

/* Get route link structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_route_link(Tgps conn, TgpsRouteLnk r)
{
  int ret;
  ret = tgps_get_and_parse_line(conn, r,
				tgps_rlnk_check_for_k1,
				tgps_rlnk_get_ident,
				tgps_rlnk_get_options,
				NULL);
  /* Return ok, even if options field is missing */
  if (ret == -2)
    {
      conn->input_buffer_len = -1;
      return 1;
    }
  return ret;
}

/* Check for t format. Return 0 in case of error, and 1 if ok. */
int tgps_trk_check_for_t(Tgps conn, TgpsTrack t, char *ptr)
{
  if (strcasecmp(ptr, "T") == 0)
    return 1;
  return 0;
}

/* Get time from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_trk_get_time(Tgps conn, void *data, char *ptr,
		      void *data_ptr, size_t data_len)
{
  TgpsTrack t = data;

  return tgps_get_time(conn, data, ptr, &t->time, sizeof(t->time));
}

/* Get latitude. Return 0 in case of error, and 1 if ok. */
int tgps_trk_get_lat(Tgps conn, TgpsTrack t, char *ptr)
{
  return tgps_get_pos(conn, ptr, &t->lat);
}

/* Get longitude. Return 0 in case of error, and 1 if ok. */
int tgps_trk_get_lon(Tgps conn, TgpsTrack t, char *ptr)
{
  return tgps_get_pos(conn, ptr, &t->lon);
}

/* Get track point structure in gpstrans format. Return 0 in case of error, and
   1 if ok. */
int tgps_get_track_gpstrans(Tgps conn, TgpsTrack t)
{
  int ret;
  ret = tgps_get_and_parse_line(conn, t,
				tgps_trk_check_for_t,
				tgps_trk_get_time,
				tgps_trk_get_lat,
				tgps_trk_get_lon,
				NULL);
  /* Check if line ended while reading first item, i.e. empty line */
  if (ret == -1)
    {
      /* Yes, this is new track */
      conn->input_buffer_len = -1;
      t->new_track = 1;
      return tgps_get_and_parse_line(conn, t,
				     tgps_trk_check_for_t,
				     tgps_trk_get_time,
				     tgps_trk_get_lat,
				     tgps_trk_get_lon,
				     NULL);
    }
  return ret;
}

/* Check for t1 format. Return 0 in case of error, and 1 if ok. */
int tgps_trk_check_for_t1(Tgps conn, TgpsTrack t, char *ptr)
{
  if (strcasecmp(ptr, "T1") == 0)
    return 1;
  return 0;
}

/* Get options. Return 0 in case of error, and 1 if ok. */
int tgps_trk_get_options(Tgps conn, TgpsTrack t, char *ptr)
{
  return
    tgps_get_and_parse_options(conn, t, ptr,
			       "newtrack=", tgps_get_boolean, &t->new_track, 0,
			       "alt=", tgps_get_float, &t->alt, 0, 
			       "depth=", tgps_get_float, &t->dpth, 0,
			       "extra=", tgps_get_hexbytes, t->extra,
			       sizeof(t->extra),
			       NULL);
}


/* Get track point structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_track_v1(Tgps conn, TgpsTrack t)
{
  int ret;
  ret = tgps_get_and_parse_line(conn, t,
				tgps_trk_check_for_t1,
				tgps_trk_get_time,
				tgps_trk_get_lat,
				tgps_trk_get_lon,
				tgps_trk_get_options,
				NULL);
  /* Return ok, even if options field is missing */
  if (ret == -4)
    {
      conn->input_buffer_len = -1;
      return 1;
    }
  return ret;
}

/* Get track point structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_track(Tgps conn, TgpsTrack t)
{
  int ret;

  ret = tgps_get_track_v1(conn, t);
  if (ret)
    return ret;
  return tgps_get_track_gpstrans(conn, t);
}

/* Check for n1 format. Return 0 in case of error, and 1 if ok. */
int tgps_trk_hdr_check_for_n1(Tgps conn, TgpsTrackHeader t, char *ptr)
{
  if (strcasecmp(ptr, "N1") == 0)
    return 1;
  return 0;
}

/* Get options. Return 0 in case of error, and 1 if ok. */
int tgps_trk_hdr_get_options(Tgps conn, TgpsTrackHeader t, char *ptr)
{
  return
    tgps_get_and_parse_options(conn, t, ptr,
			       "display=", tgps_get_boolean, &t->display, 0, 
			       "color=", tgps_get_color, &t->color, 0,
			       "width=", tgps_get_long, &t->line_width, 0,
			       "fill=", tgps_get_boolean, &t->fill, 0,
			       NULL);
}

/* Get ident string. Return 0 in case of error, and 1 if ok. */
int tgps_trk_hdr_get_ident(Tgps conn, TgpsTrackHeader t, char *ptr)
{
  t->ident = strdup(ptr);
  return 1;
}

/* Get track header structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_track_hdr(Tgps conn, TgpsTrackHeader t)
{
  int ret;
  ret = tgps_get_and_parse_line(conn, t,
				tgps_trk_hdr_check_for_n1,
				tgps_trk_hdr_get_ident,
				tgps_trk_hdr_get_options,
				NULL);
  /* Return ok, even if options field is missing */
  if (ret == -2)
    {
      conn->input_buffer_len = -1;
      return 1;
    }
  return ret;
}

/* Check for c1 format. Return 0 in case of error, and 1 if ok. */
int tgps_clock_check_for_c1(Tgps conn, TgpsTime t, char *ptr)
{
  if (strcasecmp(ptr, "C1") == 0)
    return 1;
  return 0;
}

/* Get time from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_clock_get_time(Tgps conn, void *data, char *ptr,
		       void *data_ptr, size_t data_len)
{
  TgpsTime t = data;
  int year, mon, day, hour, min, sec;

  if (sscanf(ptr, "%d-%d-%d %d:%d:%d GMT",
	     &year, &mon, &day, &hour, &min, &sec) == 7)
    {
      t->year = year;
      t->month = mon;
      t->day = day;
      t->hour = hour;
      t->minute = min;
      t->second = sec;
      return 1;
    }
  return 0;
}

/* Get time structure */
int tgps_get_clock(Tgps conn, TgpsTime t)
{
  return tgps_get_and_parse_line(conn, t,
				 tgps_clock_check_for_c1,
				 tgps_clock_get_time,
				 NULL);
}

/* Check for p1 format. Return 0 in case of error, and 1 if ok. */
int tgps_position_check_for_p1(Tgps conn, TgpsPosition p, char *ptr)
{
  if (strcasecmp(ptr, "P1") == 0)
    return 1;
  return 0;
}

/* Get latitude. Return 0 in case of error, and 1 if ok. */
int tgps_position_get_lat(Tgps conn, TgpsPosition p, char *ptr)
{
  return tgps_get_pos_double(conn, ptr, &p->lat);
}

/* Get longitude. Return 0 in case of error, and 1 if ok. */
int tgps_position_get_lon(Tgps conn, TgpsPosition p, char *ptr)
{
  return tgps_get_pos_double(conn, ptr, &p->lon);
}

/* Get position structure */
int tgps_get_position(Tgps conn, TgpsPosition p)
{
  return tgps_get_and_parse_line(conn, p,
				 tgps_position_check_for_p1,
				 tgps_position_get_lat,
				 tgps_position_get_lon,
				 NULL);
}

/* Check for l1 format. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_check_for_l1(Tgps conn, TgpsPvt p, char *ptr)
{
  if (strcasecmp(ptr, "L1") == 0)
    return 1;
  return 0;
}

/* Get latitude. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_lat(Tgps conn, TgpsPvt p, char *ptr)
{
  int ret;
  ret = tgps_get_pos_double(conn, ptr, &p->lat);
  if (ret)
    p->lat = tgps_deg_to_radians(p->lat);
  return ret;
}

/* Get longitude. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_lon(Tgps conn, TgpsPvt p, char *ptr)
{
  int ret;
  ret = tgps_get_pos_double(conn, ptr, &p->lon);
  if (ret)
    p->lon = tgps_deg_to_radians(p->lon);
  return ret;
}

/* Get alt. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_alt(Tgps conn, TgpsPvt p, char *ptr)
{
  return tgps_get_float(conn, p, ptr, &p->alt, sizeof(p->alt));
}

/* Get mean seal level. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_msl(Tgps conn, TgpsPvt p, char *ptr)
{
  return tgps_get_float(conn, p, ptr, &p->msl_hght, sizeof(p->msl_hght));
}

/* Get speed. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_speed(Tgps conn, TgpsPvt p, char *ptr)
{
  return tgps_get_float_3(conn, ptr, &p->east, &p->north, &p->up);
}

/* Get epe. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_epe(Tgps conn, TgpsPvt p, char *ptr)
{
  return tgps_get_float_3(conn, ptr, &p->epe, &p->eph, &p->epv);
}

/* Get fix. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_fix(Tgps conn, TgpsPvt p, char *ptr)
{
  int f;
  f = tgps_string_to_fix(ptr);
  if (f == -1)
    return 0;
  p->fix = f;
  return 1;
}

/* Get time. Return 0 in case of error and 1 if ok. */
int tgps_pvt_get_time(Tgps conn, TgpsPvt p, char *ptr)
{
  char *endptr;
  long l;

  l = strtol(ptr, &endptr, 0);

  /* Is it empty */
  if (ptr == endptr)
    return 0;
  while (isspace(*endptr))
    endptr++;
  /* Is there d at the end */
  if (*endptr++ != 'd')
    return 0;
  if (*endptr++ != '+')
    return 0;

  p->wn_days = l;

  ptr = endptr;
  l = strtol(ptr, &endptr, 0);

  /* Is it empty */
  if (ptr == endptr)
    return 0;
  while (isspace(*endptr))
    endptr++;
  /* Is there s at the end */
  if (*endptr++ != 's')
    return 0;
  if (*endptr++ != '-')
    return 0;

  p->tow = l;

  ptr = endptr;
  l = strtol(ptr, &endptr, 0);

  /* Is it empty */
  if (ptr == endptr)
    return 0;
  while (isspace(*endptr))
    endptr++;
  /* Is there ls at the end */
  if (*endptr++ != 'l')
    return 0;
  if (*endptr++ != 's')
    return 0;
  if (*endptr)
    return 0;

  p->leap_scnds = l;
  return 1;
}

/* Get PVT structure */
int tgps_get_pvt(Tgps conn, TgpsPvt p)
{
  int ret;
  ret = tgps_get_and_parse_line(conn, p,
				tgps_pvt_check_for_l1,
				tgps_pvt_get_lat,
				tgps_pvt_get_lon,
				tgps_pvt_get_alt,
				tgps_pvt_get_msl,
				tgps_pvt_get_speed,
				tgps_pvt_get_epe,
				tgps_pvt_get_fix,
				tgps_pvt_get_time,
				NULL);
  if (ret)
    p->alt -= p->msl_hght;
  return ret;
}

/* Almanac context */
struct TgpsAlmanacContextRec {
  TgpsAlmanac a;
  void *ptr;
  int (*get_func)(Tgps conn, void *data, char *ptr,
		  void *data_ptr, size_t data_len);
};

/* Check for a1 format. Return 0 in case of error, and 1 if ok. */
int tgps_almanac_check_for_a1(Tgps conn, TgpsAlmanacContext aa, char *ptr)
{
  if (strcasecmp(ptr, "A1") == 0)
    return 1;
  return 0;
}

/* Get satellite id. Return 0 in case of error, and 1 if ok. */
int tgps_almanac_get_svid(Tgps conn, TgpsAlmanacContext aa, char *ptr)
{
  char c;
  
  tgps_get_byte(conn, aa, ptr, &c, sizeof(c));
  if (aa->a->svid == -1)
    {
      aa->a->svid = c;
      return 1;
    }
  else if (aa->a->svid == c)
    return 1;
  return 0;
}

/* Get data item name. Return 0 in case of error, and 1 if ok. */
int tgps_almanac_get_name(Tgps conn, TgpsAlmanacContext aa, char *ptr)
{
  if (strcasecmp(ptr, "wn") == 0)
    {
      aa->ptr = &(aa->a->wn);
      aa->get_func = tgps_get_byte;
    }
  else if (strcasecmp(ptr, "toa") == 0)
    {
      aa->ptr = &(aa->a->toa);
      aa->get_func = tgps_get_float;
    }
  else if (strcasecmp(ptr, "af0") == 0)
    {
      aa->ptr = &(aa->a->af0);
      aa->get_func = tgps_get_float;
    }
  else if (strcasecmp(ptr, "af1") == 0)
    {
      aa->ptr = &(aa->a->af1);
      aa->get_func = tgps_get_float;
    }
  else if (strcasecmp(ptr, "e") == 0)
    {
      aa->ptr = &(aa->a->e);
      aa->get_func = tgps_get_float;
    }
  else if (strcasecmp(ptr, "sqrta") == 0)
    {
      aa->ptr = &(aa->a->sqrta);
      aa->get_func = tgps_get_float;
    }
  else if (strcasecmp(ptr, "m0") == 0)
    {
      aa->ptr = &(aa->a->m0);
      aa->get_func = tgps_get_float;
    }
  else if (strcasecmp(ptr, "w") == 0)
    {
      aa->ptr = &(aa->a->w);
      aa->get_func = tgps_get_float;
    }
  else if (strcasecmp(ptr, "omg0") == 0)
    {
      aa->ptr = &(aa->a->omg0);
      aa->get_func = tgps_get_float;
    }
  else if (strcasecmp(ptr, "odot") == 0)
    {
      aa->ptr = &(aa->a->odot);
      aa->get_func = tgps_get_float;
    }
  else if (strcasecmp(ptr, "i") == 0)
    {
      aa->ptr = &(aa->a->i);
      aa->get_func = tgps_get_float;
    }
  else if (strcasecmp(ptr, "hlth") == 0)
    {
      aa->ptr = &(aa->a->hlth);
      aa->get_func = tgps_get_byte;
    }
  else if (strcasecmp(ptr, "extra") == 0)
    {
      aa->ptr = &(aa->a->hlth);
      aa->get_func = tgps_get_hexbytes_3;
    }
  else
    {
      return 0;
    }
  return 1;
}

/* Get data item data. Return 0 in case of error, and 1 if ok. */
int tgps_almanac_get_data(Tgps conn, TgpsAlmanacContext aa, char *ptr)
{
   return (*(aa->get_func))(conn, aa->a, ptr, aa->ptr, 0);
}


/* Get almanac structure */
int tgps_get_almanac_v1(Tgps conn, TgpsAlmanac a)
{
  TgpsAlmanacContextStruct aa;
  int ret, cnt = 0;

  aa.a = a;
  aa.a->svid = -1;
  do {
    ret = tgps_get_and_parse_line(conn, &aa,
				  tgps_almanac_check_for_a1,
				  tgps_almanac_get_svid,
				  tgps_almanac_get_name,
				  tgps_almanac_get_data,
				  NULL);
    if (ret == -2)
      continue;
    if (ret == 1)
      cnt++;
  } while (ret == 1);
  if (cnt == 12)
    return 1;
  return 0;
}  
  
/* Get almanac structure */
int tgps_get_almanac(Tgps conn, TgpsAlmanac a)
{
  return tgps_get_almanac_v1(conn, a);
}

#if 0
/* Get header */
int tgps_get_header_v1(Tgps conn)
{
}

/* Get header in gpstrans format */
int tgps_get_header_gpstrans(Tgps conn)
{
}

/* Get header */
int tgps_get_header(Tgps conn)
{
}

#endif

