/*
 * tgps.c -- GPS protocol main.
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/tgps.c,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 15:13 Apr  7 2000 kivinen
 *	  Last Modification : 11:00 Aug 19 2003 kivinen
 *	  Last check in     : $Date: 2003/08/18 20:12:28 $
 *	  Revision number   : $Revision: 1.23 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.1954
 *	  Edit time	    : 963 min
 *
 *	  Description       : GPS protocol main program.
 *
 *	  $Log: tgps.c,v $
 *	  Revision 1.23  2003/08/18 20:12:28  kivinen
 *	  	Added -C option (ignore errors), use 1 byte ack/nak messages
 *	  	for most devices. Added -F option.
 *
 *	  Revision 1.22  2001/08/23 15:38:57  kivinen
 *	  	Fixed bug in tgps_track_hdr handling.
 *
 *	  Revision 1.21  2000/10/02 15:43:42  kivinen
 *	  	Changed to use tgps_print.
 *
 *	  Revision 1.20  2000/08/18 13:15:13  kivinen
 *	  	Fixed bug.
 *
 *	  Revision 1.19  2000/08/05 00:44:16  kivinen
 *	  	Updated to new interface.
 *
 *	  Revision 1.18  2000/07/26 17:10:43  kivinen
 *	  	Changed records and xfer complete so that they are not
 *	  	cumulative. Added position, time and almanac upload support.
 *
 *	  Revision 1.17  2000/07/21 22:17:06  kivinen
 *	  	Copied stuff from input.c. Moved stuff to util.c.
 *
 *	  Revision 1.16  2000/07/17 21:39:35  kivinen
 *	  	Fixed typo in linux serial fix.
 *
 *	  Revision 1.15  2000/07/17 21:34:30  kivinen
 *	  	Added retry count and linux serial fix kludge.
 *
 *	  Revision 1.14  2000/07/15 19:53:31  kivinen
 *	  	Added printing of number of received data packets after the
 *	  	transfer is completed. Changed resending message to be printed
 *	  	only if verbose level is 4 or more. Added time to PVT active,
 *	  	but nothing received message.
 *
 *	  Revision 1.13  2000/07/12 22:20:18  kivinen
 *	  	Final upload support.
 *
 *	  Revision 1.12  2000/07/11 20:12:47  kivinen
 *	  	Fixed bug in abort operation.
 *
 *	  Revision 1.11  2000/07/10 21:39:25  kivinen
 *	  	Added more option checks. Some other small changes.
 *
 *	  Revision 1.10  2000/07/10 19:32:17  kivinen
 *	  	Fixed some bugs.
 *
 *	  Revision 1.9  2000/07/08 21:17:13  kivinen
 *	  	Fixed bug in duplicate packets being acked.
 *
 *	  Revision 1.8  2000/07/08 20:45:57  kivinen
 *	  	Modified error texts. Added -v option. Now it does not exit
 *	  	after receiving something if no operation is not given.
 *	  	Removed converting slash to dash in the output file name.
 *	  	Fixed some bugs.
 *
 *	  Revision 1.7  2000/07/08 17:38:44  kivinen
 *	  	Added support for strftime of output filename. Do not print
 *	  	verbose output if stderr is not tty.
 *
 *	  Revision 1.6  2000/07/07 20:41:11  kivinen
 *	  	Added support for ignoring compressed tracks. Added linux
 *	  	support.
 *
 *	  Revision 1.5  2000/07/06 23:05:11  kivinen
 *	  	Added initial upload support.
 *
 *	  Revision 1.4  2000/04/30 01:37:34  kivinen
 *	  	Updated usage.
 *
 *	  Revision 1.3  2000/04/30 01:33:25  kivinen
 *	  	Fixed memory allocation bug.
 *
 *	  Revision 1.2  2000/04/30 01:03:44  kivinen
 *	  	Updated to rev 03 document.
 *
 *	  Revision 1.1  2000/04/29 16:40:37  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#include "tgps.h"
#include "data.h"
#include "display.h"
#include "packet.h"
#include "ieee-double.h"
#include "input.h"

/* Open the serial line */
int tgps_serial_open(Tgps conn, char *tty_name);

/* Open connection to gps */
Tgps tgps_open(char *tty_name, char *output_file, char *input_file);

/* Close connection to gps */
int tgps_close(Tgps conn);

/* Read data from gps */
int tgps_read(Tgps conn);

/* Write data to gps */
int tgps_write(Tgps conn);

/* Wait data to be available from gps, if secs and usecs are 0, then do not set
   timeout. */
int tgps_wait(Tgps conn, int secs, int usecs);

/* Find packet from the input buffer */
TgpsPacket tgps_find_packet(Tgps conn);

/* Add packet to output buffer */
void tgps_add_packet(Tgps conn, TgpsPacket packet);

/* Add byte to the packet */
void tgps_add_byte(TgpsPacket packet, unsigned char byte);

/* Add quoted byte to packet */
void tgps_add_quoted_byte(TgpsPacket packet, unsigned char byte);

/* Send packet, add it to the commands[] structure if not ACK or NAK packet */
int tgps_send_packet(Tgps conn, unsigned char command,
		     unsigned char *data, size_t len);

/* Send packet, add it to the commands[] structure if not ACK or NAK packet */
int tgps_send_complete_packet(Tgps conn, TgpsPacket p);

/* Encode and send packet, add it to the commands[] structure if not ACK or NAK
   packet */
int tgps_encode_send_packet(Tgps conn, unsigned char command, ...);

/* Send ack packet */
int tgps_send_ack(Tgps conn, unsigned char command);

/* Send nak packet */
int tgps_send_nak(Tgps conn, unsigned char command);

/* Process ack packet */
void tgps_ack_byte(Tgps conn, TgpsPacket packet);

/* Process nak packet */
void tgps_nak_byte(Tgps conn, TgpsPacket packet);

/* Print protocol information */
void tgps_print_protocol_ids(char *type, int *ids, int cnt);

/* Process product data packet */
void tgps_product_data_in(Tgps conn, TgpsPacket packet);

/* Process protocol array packet */
void tgps_protocol_array(Tgps conn, TgpsPacket packet);

/* Process records packet */
void tgps_records(Tgps conn, TgpsPacket packet);

/* Process transfer completed packet */
void tgps_xfer_cmplt(Tgps conn, TgpsPacket packet);

/* Process packet */
void tgps_process_packet(Tgps conn, TgpsPacket packet);

/* Signal handler that will abort the transfer */
void tgps_abort_transfer(int sig);

/* Send abort operation to gps. */
void tgps_abort(Tgps conn);

/* Start transfer */
void tgps_start_transfer(Tgps conn, int command);

/* Get line */
TgpsPacket tgps_get_line(Tgps conn);

#define TGPS_PACKET_DUMP

int tgps_packet_dump = 0;

/* Check if the protocol version is supported */
int tgps_is_supported(Tgps conn, unsigned char type, int version)
{
  int *table;
  int cnt, i;

  switch (type)
    {
    case 'P':
      table = conn->phys_prot_ids; cnt = conn->phys_prot_id_cnt;
      break;
    case 'L':
      table = conn->link_prot_ids; cnt = conn->link_prot_id_cnt;
      break;
    case 'A':
      table = conn->appl_prot_ids; cnt = conn->appl_prot_id_cnt;
      break;
    case 'D':
      table = conn->data_prot_ids; cnt = conn->data_prot_id_cnt;
      break;
    default:
      fprintf(stderr, "Internal Error: unknown protocol type : %c (%02x)\n",
	      type, type);
      exit(1);
      break;
    }

  for(i = 0; i < cnt; i++)
    {
      if (table[i] == version)
	return 1;
    }
  return 0;
}

/* Open the serial line */
int tgps_serial_open(Tgps conn, char *tty_name)
{
  struct termios t;

  conn->tty = open(tty_name, O_RDWR | O_NONBLOCK, 0666);
  if (conn->tty < 0)
    {
      perror("open tty failed");
      return 0;
    }

  if (tcgetattr(conn->tty, &conn->orig_tty) < 0)
    {
      perror("tcgetattr failed");
      return 0;
    }

#ifdef VEOF
  t.c_cc[VEOF] = _POSIX_VDISABLE;
#endif /* VEOF */
#ifdef VEOL
    t.c_cc[VEOL] = _POSIX_VDISABLE;
#endif /* VEOL */
#ifdef VEOL2
    t.c_cc[VEOL2] = _POSIX_VDISABLE;
#endif /* VEOL2 */
#ifdef VERASE
    t.c_cc[VERASE] = _POSIX_VDISABLE;
#endif /* VERASE */
#ifdef VWERASE
    t.c_cc[VWERASE] = _POSIX_VDISABLE;
#endif /* VWERASE */
#ifdef VKILL
    t.c_cc[VKILL] = _POSIX_VDISABLE;
#endif /* VKILL */
#ifdef VREPRINT
    t.c_cc[VREPRINT] = _POSIX_VDISABLE;
#endif /* VREPRINT */
#ifdef VINTR
    t.c_cc[VINTR] = _POSIX_VDISABLE;
#endif /* VINTR */
#ifdef VQUIT
    t.c_cc[VQUIT] = _POSIX_VDISABLE;
#endif /* VQUIT */
#ifdef VSUSP
    t.c_cc[VSUSP] = _POSIX_VDISABLE;
#endif /* VSUSP */
#ifdef VDSUSP
    t.c_cc[VDSUSP] = _POSIX_VDISABLE;
#endif /* VDSUSP */
#ifdef VSTART
    t.c_cc[VSTART] = _POSIX_VDISABLE;
#endif /* VSTART */
#ifdef VSTOP
    t.c_cc[VSTOP] = _POSIX_VDISABLE;
#endif /* VSTOP */
#ifdef VLNEXT
    t.c_cc[VLNEXT] = _POSIX_VDISABLE;
#endif /* VLNEXT */
#ifdef VDISCARD
    t.c_cc[VDISCARD] = _POSIX_VDISABLE;
#endif /* VDISCARD */
#ifdef VSTATUS
    t.c_cc[VSTATUS] = _POSIX_VDISABLE;
#endif /* VSTATUS */
#ifdef VMIN
    t.c_cc[VMIN] = 0;
#endif /* VMIN */
#ifdef VTIME
    t.c_cc[VTIME] = 1;
#endif /* VTIME */

  t.c_cflag = CS8 | CREAD | CLOCAL;
  t.c_iflag = IGNBRK | IGNPAR;
  t.c_oflag = 0;
#ifdef NOKERNINFO
  t.c_lflag = NOKERNINFO;
#else 
  t.c_lflag = 0;
#endif /* NOKERNINFO */

  if (cfsetispeed(&t, B9600) < 0)
    {
      perror("cfsetispeed failed");
      return 0;
    }

  if (cfsetospeed(&t, B9600) < 0)
    {
      perror("cfsetospeed failed");
      return 0;
    }

  if (tcsetattr(conn->tty, TCSANOW, &t) < 0)
    {
      perror("tcsetattr failed");
      return 0;
    }
  return 1;
}

/* Open connection to gps */
Tgps tgps_open(char *tty_name, char *output_file, char *input_file)
{
  Tgps conn;
  int i;

  conn = calloc(1, sizeof(*conn));
  if (conn == NULL)
    return NULL;

  if (tgps_serial_open(conn, tty_name) == 0)
    {
      close(conn->tty);
      free(conn);
      return NULL;
    }

  if (output_file == NULL || strcmp(output_file, "-") == 0)
    conn->output_stream = stdout;
  else
    {
#ifdef HAVE_STRFTIME
      char buffer[512];
      struct tm *tptr;
      time_t t;

      t = time(NULL);
      tptr = localtime(&t);
      strftime(buffer, sizeof(buffer), output_file, tptr);
      conn->output_stream = fopen(buffer, "ab");
#else /* HAVE_STRFTIME */
      conn->output_stream = fopen(output_file, "ab");
#endif /* HAVE_STRFTIME */
    }

  if (conn->output_stream == NULL)
    {
      perror("Could not open output file");
      tcsetattr(conn->tty, TCSANOW, &conn->orig_tty);
      close(conn->tty);
      free(conn);
      return NULL;
    }

  if (input_file == NULL || strcmp(input_file, "-") == 0)
    conn->input_stream = stdin;
  else
    conn->input_stream = fopen(input_file, "rb");
  if (conn->input_stream == NULL)
    {
      perror("Could not open input file");
      if (conn->output_stream != stdout)
	fclose(conn->output_stream);
      tcsetattr(conn->tty, TCSANOW, &conn->orig_tty);
      close(conn->tty);
      free(conn);
      return NULL;
    }

  conn->buffer_in_alloc_size = BUFLEN;
  conn->buffer_in_start = 0;
  conn->buffer_in_len = 0;
  conn->buffer_in = malloc(conn->buffer_in_alloc_size);
  if (conn->buffer_in == NULL)
    {
      if (conn->output_stream != stdout)
	fclose(conn->output_stream);
      if (conn->input_stream != stdin)
	fclose(conn->input_stream);
      tcsetattr(conn->tty, TCSANOW, &conn->orig_tty);
      close(conn->tty);
      free(conn);
      return NULL;
    }

  conn->buffer_out_alloc_size = BUFLEN;
  conn->buffer_out_start = 0;
  conn->buffer_out_len = 0;
  conn->buffer_out = malloc(conn->buffer_out_alloc_size);
  if (conn->buffer_out == NULL)
    {
      if (conn->output_stream != stdout)
	fclose(conn->output_stream);
      if (conn->input_stream != stdin)
	fclose(conn->input_stream);
      tcsetattr(conn->tty, TCSANOW, &conn->orig_tty);
      close(conn->tty);
      free(conn->buffer_in);
      free(conn);
      return NULL;
    }

  conn->buffer_last_packet = malloc(256);
  if (conn->buffer_last_packet == NULL)
    {
      if (conn->output_stream != stdout)
	fclose(conn->output_stream);
      if (conn->input_stream != stdin)
	fclose(conn->input_stream);
      tcsetattr(conn->tty, TCSANOW, &conn->orig_tty);
      close(conn->tty);
      free(conn->buffer_in);
      free(conn);
      return NULL;
    }
  conn->buffer_last_packet_len = 0;
  conn->free_list = NULL;
  for(i = 0; i < 256; i++)
    conn->command[i] = NULL;
  conn->product_id = 0;
  conn->software_version = 0;
  conn->phys_prot_ids = NULL;
  conn->phys_prot_id_cnt = 0;
  conn->link_prot_ids = NULL;
  conn->link_prot_id_cnt = 0;
  conn->appl_prot_ids = NULL;
  conn->appl_prot_id_cnt = 0;
  conn->data_prot_ids = NULL;
  conn->data_prot_id_cnt = 0;
  conn->packet_count = 0;
  conn->packet_count_total = 0;
  conn->xfer_done = 0;
  conn->degree_format = TGPS_DEG;
  conn->data_version = TGPS_DATA_VERSION_TGPS_V1;
  conn->pvt_active = 0;

  conn->input_buffer_len = -1;
  conn->input_buffer_alloc = 1024;
  conn->input_buffer = malloc(conn->input_buffer_alloc);
  if (conn->input_buffer == NULL)
    {
      if (conn->output_stream != stdout)
	fclose(conn->output_stream);
      if (conn->input_stream != stdin)
	fclose(conn->input_stream);
      tcsetattr(conn->tty, TCSANOW, &conn->orig_tty);
      close(conn->tty);
      free(conn->buffer_in);
      free(conn->buffer_out);
      free(conn->buffer_last_packet);
      free(conn);
      return NULL;
    }
  return conn;
}

/* Close connection to gps */
int tgps_close(Tgps conn)
{
  TgpsPacket packet, next;
  int i;

  if (tcsetattr(conn->tty, TCSANOW, &conn->orig_tty) < 0)
    {
      perror("tcsetattr failed");
      free(conn);
      return -1;
    }
  if (conn->output_stream != stdout)
    fclose(conn->output_stream);
  if (conn->input_stream != stdin)
    fclose(conn->input_stream);
  close(conn->tty);
  free(conn->buffer_in);
  free(conn->buffer_out);
  free(conn->buffer_last_packet);
  free(conn->input_buffer);
  for(packet = conn->free_list; packet != NULL; packet = next)
    {
      next = packet->next;
      free(packet->data);
      free(packet);
    }

  for(i = 0; i < 256; i++)
    {
      for(packet = conn->command[i]; packet != NULL; packet = next)
	{
	  next = packet->next;
	  free(packet->data);
	  free(packet);
	}
    }
  free(conn->product_description);
  free(conn->phys_prot_ids);
  free(conn->link_prot_ids);
  free(conn->appl_prot_ids);
  free(conn->data_prot_ids);
  free(conn);
  return 0;
}

/* Read data from gps */
int tgps_read(Tgps conn)
{
  int ret;

  if (conn->buffer_in_start + conn->buffer_in_len ==
      conn->buffer_in_alloc_size)
    {
      if (conn->buffer_in_start == 0)
	{
	  fprintf(stderr, "Error: Buffer overflow in tgps_read\n");
	  conn->buffer_in_len =  0;
	}
      else
	{
	  memmove(conn->buffer_in, conn->buffer_in + conn->buffer_in_start,
		  conn->buffer_in_len);
	  conn->buffer_in_start = 0;
	}
    }
  ret = read(conn->tty, conn->buffer_in + conn->buffer_in_start +
	     conn->buffer_in_len, conn->buffer_in_alloc_size -
	     conn->buffer_in_len - conn->buffer_in_start);
  if (ret < 0)
    {
      if (errno == EAGAIN)
	return 0;
      perror("Read failed in tgps_read");
      return -1;
    }
#ifdef TGPS_PACKET_DUMP
#if 0
  if (tgps_packet_dump)
    tgps_print_buffer("Read: ", conn->buffer_in + conn->buffer_in_start +
		      conn->buffer_in_len, ret);
#endif
#endif
  conn->buffer_in_len += ret;
  return 1;
}

/* Write data to gps */
int tgps_write(Tgps conn)
{
  int ret;

  while (conn->buffer_out_len > 0)
    {
      ret = write(conn->tty, conn->buffer_out + conn->buffer_out_start,
		  conn->buffer_out_len);
      if (ret < 0)
	{
	  if (errno == EAGAIN)
	    return 0;
	  perror("Write failed in tgps_write");
	  return -1;
	}

      if (ret == 0)
	return 0;
#ifdef TGPS_PACKET_DUMP
      if (tgps_packet_dump)
	tgps_print_buffer("Writing: ",
			  conn->buffer_out + conn->buffer_out_start,
			  ret);
#endif

      conn->buffer_out_start += ret;
      conn->buffer_out_len -= ret;
    }
  conn->buffer_out_start = 0;
  return 0;
}

/* Wait data to be available from gps, if secs and usecs are 0, then do not set
   timeout. */
int tgps_wait(Tgps conn, int secs, int usecs)
{
  struct timeval t, *tp;
  fd_set fdset;
  int ret;

  tp = NULL;
  if (secs != 0 || usecs != 0)
    {
      tp = &t;
      t.tv_sec = secs;
      t.tv_usec = usecs;
    }
  FD_ZERO(&fdset);
  FD_SET(conn->tty, &fdset);

  if (conn->buffer_out_len > 0)
    ret = select(conn->tty + 1, &fdset, &fdset, NULL, tp);
  else  
    ret = select(conn->tty + 1, &fdset, NULL, NULL, tp);

  if ((ret < 0 && errno == EINTR) ||
      ret == 0)
    return 0;

  if (ret < 0)
    {
      perror("select failed in tgps_wait");
      return -1;
    }
  return 1;
}

/* Find packet from the input buffer */
TgpsPacket tgps_find_packet(Tgps conn)
{
  unsigned char *buffer, *p;
  size_t i, start, end, len;
  unsigned char check_sum;
  TgpsPacket packet;

  buffer = conn->buffer_in + conn->buffer_in_start;
  start = -1;
  end = -1;

  for(i = 0; i < conn->buffer_in_len; i++)
    {
      /* Possible start or end of of packet */
      if (buffer[i] == TGPS_DLE)
	{
	  /* Check if there is one more character after that */
	  if (i + 1 >= conn->buffer_in_len)
	    return NULL;

	  /* Check if it is another DLE, if so this is part of data */
	  if (buffer[i + 1] == TGPS_DLE)
	    {
	      /* Yes, this is just DLE inside the data, just ignore */
	    }
	  else if (buffer[i + 1] == TGPS_ETX) /* Check for end of the packet */
	    {
	      /* This is the end of the packet */
	      end = i + 2;
	    }
	  else /* This must be beginning of the packet */
	    {
	      if (start == -1 ||
		  !(conn->flags & TGPS_FLAGS_IGNORE_ERRORS))
		{
		  start = i;
		  end = -1;	/* Start searching for the end again */
		}
	    }
	  i++;			/* Skip the DLE character */

	  /* Do we have complete packet */
	  if (start != -1 && end != -1)
	    break;
	}
    }

  /* Check if we have packet */
  if (start == -1 || end == -1)
    {
      /* No, did we find the end of packet */
      if (end != -1)
	{
	  /* Yes, remove that partial packet from the buffer */
	  conn->buffer_in_start += end;
	  conn->buffer_in_len -= end;
	}
      else if (start != -1)
	{
	  /* We found start of the packet, but we didn't find end, remove
	     everything before the start of the packet */
	  conn->buffer_in_start += start;
	  conn->buffer_in_len -= start;
	}
      return NULL;
    }

  len = end - start - 4;
  buffer = conn->buffer_in + conn->buffer_in_start + start;

  /* Remove the packet from the buffer */
  conn->buffer_in_len -= end;
  conn->buffer_in_start += end;

  if (conn->buffer_last_packet_len == len + 4 &&
      memcmp(buffer, conn->buffer_last_packet, len + 4) == 0)
    {
      if (conn->verbose)
	fprintf(stderr, "Warning: Duplicate packet received, ignored\n");
      return NULL;
    }
  if (len + 4 > 256)
    {
      if (conn->verbose)
	fprintf(stderr, "Warning: Too big packet received, len = %d\n",
		len + 4);
      return NULL;
    }
  memcpy(conn->buffer_last_packet, buffer, len + 4);
  conn->buffer_last_packet_len = len + 4;

#ifdef TGPS_PACKET_DUMP
  if (tgps_packet_dump)
    tgps_print_buffer("Packet: ", buffer, len + 4);
#endif

  packet = tgps_get_packet(conn, len + 1);

  p = packet->data;
  packet->data_len = len + 1;
  check_sum = 0;
  buffer++;
  for(i = 0; i <= len; i++)
    {
      if (buffer[i] == TGPS_DLE)
	{
	  i++;
	  packet->data_len--;
	}
      *p++ = buffer[i];
      check_sum += buffer[i];
    }
  packet->data_len--;
  if (check_sum != 0)
    {
      fprintf(stderr, "Warning: Invalid checksum 0x%02x should be 0x%02x\n",
	      0x100 - check_sum, packet->data[packet->data_len + 1]);
      if (!(conn->flags & TGPS_FLAGS_IGNORE_ERRORS))
	{
	  tgps_free_packet(conn, packet);
	  return NULL;
	}
    }
  return packet;
}

/* Add packet to output buffer */
void tgps_add_packet(Tgps conn, TgpsPacket packet)
{
  while (conn->buffer_out_start + conn->buffer_out_len + packet->data_len >=
	 conn->buffer_out_alloc_size)
    {
      if (conn->buffer_out_start == 0)
	{
	  /* Buffer too small, enlarge it */
	  conn->buffer_out_alloc_size += BUFLEN;
	  conn->buffer_out = realloc(conn->buffer_out,
				     conn->buffer_out_alloc_size);
	  if (conn->buffer_out == NULL)
	    {
	      fprintf(stderr, "Error: Out of memory when reallocating "
		      "output buffer to %ld bytes\n",
		      (unsigned long) conn->buffer_out_alloc_size);
	      exit(1);
	    }
	}
      else
	{
	  /* Move stuff in buffer to beginning */
	  memmove(conn->buffer_out, conn->buffer_out + conn->buffer_out_start,
		  conn->buffer_out_len);
	  conn->buffer_out_start = 0;
	}
    }
  memcpy(conn->buffer_out + conn->buffer_out_start + conn->buffer_out_len,
	 packet->data, packet->data_len);
  conn->buffer_out_len += packet->data_len;
  tgps_write(conn);
}

/* Add byte to the packet */
void tgps_add_byte(TgpsPacket packet, unsigned char byte)
{
  if (packet->data_len == packet->alloc_size)
    {
      fprintf(stderr, "Internal Error: in tgps_add_byte");
      exit(1);
    }
  packet->data[packet->data_len] = byte;
  packet->data_len++;
}

/* Add quoted byte to packet */
void tgps_add_quoted_byte(TgpsPacket packet, unsigned char byte)
{
  tgps_add_byte(packet, byte);
  if (byte == TGPS_DLE)
    tgps_add_byte(packet, byte);
}

/* Send packet, add it to the commands[] structure if not ACK or NAK packet */
int tgps_send_packet(Tgps conn, unsigned char command,
		     unsigned char *data, size_t len)
{
  TgpsPacket packet, *prev;
  unsigned char check_sum;
  size_t i;
  int dles;

  dles = 0;
  for(i = 0; i < len; i++)
    if (data[i] == TGPS_DLE)
      dles++;

  packet = tgps_get_packet(conn, 3 + len + 2 + 2 + dles);
  if (packet == NULL)
    return 0;

  check_sum = 0;
  tgps_add_byte(packet, TGPS_DLE);
  tgps_add_byte(packet, command);
  check_sum -= command;
  tgps_add_quoted_byte(packet, len);
  check_sum -= len;

  for(i = 0; i < len; i++)
    {
      tgps_add_quoted_byte(packet, data[i]);
      check_sum -= data[i];
    }
  tgps_add_quoted_byte(packet, check_sum);
  tgps_add_byte(packet, TGPS_DLE);
  tgps_add_byte(packet, TGPS_ETX);

  /* Check if it is packet that needs to be acked */
  if (command != TGPS_PID_L000_ACK_BYTE &&
      command != TGPS_PID_L000_NAK_BYTE &&
      command != TGPS_PID_L001_PVT_DATA)
    {
      /* Yes, check if there is similar packet out, if not so, send this packet
	 out */
      if (conn->command[command] == NULL)
	tgps_add_packet(conn, packet);

      /* Otherwise just store it to the end of the list */
      prev = &(conn->command[command]);
      while (*prev != NULL)
	{
	  prev = &((*prev)->next);
	}
      packet->next = NULL;
      *prev = packet;
    }
  else
    {
      tgps_add_packet(conn, packet);
    }
  return 1;
}

/* Send packet, add it to the commands[] structure if not ACK or NAK packet */
int tgps_send_complete_packet(Tgps conn, TgpsPacket p)
{
  TgpsPacket packet, *prev;
  unsigned char check_sum;
  size_t i;
  int dles;

  dles = 0;
  for(i = 0; i < p->data_len; i++)
    if (p->data[i] == TGPS_DLE)
      dles++;

  packet = tgps_get_packet(conn, 1 + p->data_len + 2 + 2 + dles);
  if (packet == NULL)
    return 0;

  check_sum = 0;
  tgps_add_byte(packet, TGPS_DLE);
  tgps_add_byte(packet, p->data[0]);
  check_sum -= p->data[0];
  tgps_add_quoted_byte(packet, p->data[1] - 2);
  check_sum -= p->data[1] - 2;

  for(i = 2; i < p->data_len; i++)
    {
      tgps_add_quoted_byte(packet, p->data[i]);
      check_sum -= p->data[i];
    }
  tgps_add_quoted_byte(packet, check_sum);
  tgps_add_byte(packet, TGPS_DLE);
  tgps_add_byte(packet, TGPS_ETX);

  /* Check if it is packet that needs to be acked */
  if (p->data[0] != TGPS_PID_L000_ACK_BYTE &&
      p->data[0] != TGPS_PID_L000_NAK_BYTE &&
      p->data[0] != TGPS_PID_L001_PVT_DATA)
    {
      /* Yes, check if there is similar packet out, if not so, send this packet
	 out */
      if (conn->command[p->data[0]] == NULL)
	tgps_add_packet(conn, packet);

      /* Otherwise just store it to the end of the list */
      prev = &(conn->command[p->data[0]]);
      while (*prev != NULL)
	{
	  prev = &((*prev)->next);
	}
      packet->next = NULL;
      *prev = packet;
    }
  else
    {
      tgps_add_packet(conn, packet);
    }
  return 1;
}

/* Encode and send packet, add it to the commands[] structure if not ACK or NAK
   packet */
int tgps_encode_send_packet(Tgps conn, unsigned char command, ...)
{
  unsigned char data[256];
  size_t data_len;
  va_list ap;
  int ret;

  va_start(ap, command);
  data_len = tgps_encode_va(data, sizeof(data), ap);
  va_end(ap);
  if (data_len == -1)
    return 0;
  ret = tgps_send_packet(conn, command, data, data_len);
  return ret;
}

/* Send ack packet */
int tgps_send_ack(Tgps conn, unsigned char command)
{
  unsigned char buf[2];

  buf[0] = command;
  buf[1] = 0;
  return tgps_send_packet(conn, TGPS_PID_L000_ACK_BYTE, buf,
			  conn->flags & TGPS_FLAGS_USE_1_BYTE_ACK ? 1 : 2);
}

/* Send nak packet */
int tgps_send_nak(Tgps conn, unsigned char command)
{
  unsigned char buf[2];

  buf[0] = command;
  buf[1] = 0;
  return tgps_send_packet(conn, TGPS_PID_L000_NAK_BYTE, buf,
			  conn->flags & TGPS_FLAGS_USE_1_BYTE_ACK ? 1 : 2);
}

/* Process ack packet */
void tgps_ack_byte(Tgps conn, TgpsPacket packet)
{
  int command;

  command = packet->data[2];
  if (packet->data_len == 3 || packet->data_len == 4)
    {
      TgpsPacket old_packet;

      old_packet = conn->command[command];
      if (old_packet == NULL)
	{
	  fprintf(stderr, "Warning: Ack to unknown packet of type %d\n",
		  command);
	}
      else
	{
	  if (conn->upload & TGPS_UPLOAD_WAITING_FOR_ACK)
	    conn->upload &= ~TGPS_UPLOAD_WAITING_FOR_ACK;
	  conn->command[command] = old_packet->next;
	  old_packet->next = NULL;
	  tgps_free_packet(conn, old_packet);

	  /* Check if there is next packet, and if so, send it */
	  if (conn->command[command] != NULL)
	    tgps_add_packet(conn, conn->command[command]);
	}
    }
  else
    {
      fprintf(stderr, "Warning: Invalid ack packet, data "
	      "size not 1 or 2, size = %d\n",
	      packet->data_len - 2);
    }
}

/* Process nak packet */
void tgps_nak_byte(Tgps conn, TgpsPacket packet)
{
  int command;

  command = packet->data[2];
  if (packet->data_len == 3 || packet->data_len == 4)
    {
      TgpsPacket old_packet;

      old_packet = conn->command[command];
      if (old_packet == NULL)
	{
	  fprintf(stderr, "Warning: Nak to unknown packet of type %d\n",
		  command);
	}
      else
	{
	  tgps_add_packet(conn, old_packet);
	}
    }
  else
    {
      fprintf(stderr, "Warning: Invalid nak packet, data "
	      "size not 1 or 2, size = %d\n",
	      packet->data_len - 2);
    }
}

/* Print protocol information */
void tgps_print_protocol_ids(char *type, int *ids, int cnt)
{
  int i;

  for(i = 0; i < cnt; i++)
    fprintf(stderr, "%s%03d ", type, ids[i]);
}

/* Process product data packet */
void tgps_product_data_in(Tgps conn, TgpsPacket packet)
{
  size_t size, pos;

  free(conn->product_description);
  conn->product_description = NULL;
  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_INT, conn->product_id != 0 ? NULL : &conn->product_id,
		     TGPS_FORMAT_INT, &conn->software_version,
		     TGPS_FORMAT_STRING, &conn->product_description,
		     TGPS_FORMAT_END);
  if (size == -1)
    {
      fprintf(stderr, "Error: decoding product data packet");
      return;
    }
  if (conn->verbose)
    {
      fprintf(stderr, "Product id = %s (%d), software version = %d.%02d\n"
	      "description = %s\n",
	      tgps_find_garmin_name(conn->product_id),
	      conn->product_id,
	      conn->software_version / 100,
	      conn->software_version % 100,
	      conn->product_description);
    }
  tgps_find_phys_protocol_ids(conn->product_id, conn->software_version, 
			      &conn->phys_prot_ids, &conn->phys_prot_id_cnt);
  tgps_find_link_protocol_ids(conn->product_id, conn->software_version, 
			      &conn->link_prot_ids, &conn->link_prot_id_cnt);
  tgps_find_appl_protocol_ids(conn->product_id, conn->software_version, 
			      &conn->appl_prot_ids, &conn->appl_prot_id_cnt);
  tgps_find_data_protocol_ids(conn->product_id, conn->software_version, 
			      &conn->data_prot_ids, &conn->data_prot_id_cnt);
  if (conn->verbose)
    {
      fprintf(stderr, "Supported protocols from internal tables: ");
      tgps_print_protocol_ids("P", conn->phys_prot_ids,
			      conn->phys_prot_id_cnt);
      tgps_print_protocol_ids("L", conn->link_prot_ids,
			      conn->link_prot_id_cnt);
      tgps_print_protocol_ids("A", conn->appl_prot_ids,
			      conn->appl_prot_id_cnt);
      tgps_print_protocol_ids("D", conn->data_prot_ids,
			      conn->data_prot_id_cnt);
      fprintf(stderr, "\n");

      if (packet->data_len > size)
	{
	  unsigned char *str, *p;

	  fprintf(stderr, "Additional strings: \n");
	  pos = size;
	  do {
	    size = tgps_decode(packet->data + pos, packet->data_len,
			       TGPS_FORMAT_STRING, &str,
			       TGPS_FORMAT_END);
	    if (size == -1)
	      break;
	    for(p = str; *p; p++)
	      if (iscntrl(*p))
		break;
	    if (!*p)
	      fprintf(stderr, "%s\n", str);
	    else
	      tgps_print_buffer(NULL, str, strlen(str));
	    pos += size;
	  } while (packet->data_len > pos);
	}
    }
}

/* Process protocol array packet */
void tgps_protocol_array(Tgps conn, TgpsPacket packet)
{
  int i, j;
  int **table;
  int *cnt;

  conn->phys_prot_id_cnt = 0;
  conn->link_prot_id_cnt = 0;
  conn->appl_prot_id_cnt = 0;
  conn->data_prot_id_cnt = 0;

  if (conn->flags & TGPS_FLAGS_IGNORE_ERRORS)
    return;
  for(i = 2; i + 3 <= packet->data_len; i += 3)
    {
      char type;
      int version;

      tgps_decode(packet->data + i, 3,
		  TGPS_FORMAT_BYTE, &type,
		  TGPS_FORMAT_INT, &version,
		  TGPS_FORMAT_END);

      table = NULL;
      switch (type)
	{
	case 'P':
	  table = &conn->phys_prot_ids; cnt = &conn->phys_prot_id_cnt;
	  break;
	case 'L':
	  table = &conn->link_prot_ids; cnt = &conn->link_prot_id_cnt;
	  break;
	case 'A':
	  table = &conn->appl_prot_ids; cnt = &conn->appl_prot_id_cnt;
	  break;
	case 'D':
	  table = &conn->data_prot_ids; cnt = &conn->data_prot_id_cnt;
	  break;
	case 'T':
	  /* Garmin GPS V seems to be sending this, do not know what it is.
	     ignore now. */
	  break;
	default:
	  fprintf(stderr, "Error: Unknown protocol type : %c (%02x)\n",
		  type, type);
	  break;
	}
      if (table != NULL)
	{
	  for(j = 0; j < *cnt; j++)
	    {
	      if ((*table)[j] == version)
		break;
	    }
	  if (j == *cnt)
	    {
	      *table = realloc(*table, (*cnt + 1) * sizeof(int));
	      if (*table == NULL)
		{
		  fprintf(stderr, "Error: Out of memory in "
			  "tgps_protocol_array\n");
		  exit(1);
		}
	      (*table)[*cnt] = version;
	      (*cnt)++;
	    }
	}
    }
  if (conn->verbose)
    {
      fprintf(stderr, "Supported protocols reported by GPS: ");
      tgps_print_protocol_ids("P", conn->phys_prot_ids,
			      conn->phys_prot_id_cnt);
      tgps_print_protocol_ids("L", conn->link_prot_ids,
			      conn->link_prot_id_cnt);
      tgps_print_protocol_ids("A", conn->appl_prot_ids,
			      conn->appl_prot_id_cnt);
      tgps_print_protocol_ids("D", conn->data_prot_ids,
			      conn->data_prot_id_cnt);
      fprintf(stderr, "\n");
    }      
}

/* Process records packet */
void tgps_records(Tgps conn, TgpsPacket packet)
{
  size_t size;
  int count;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_INT, &count,
		     TGPS_FORMAT_END);

  if (conn->verbose)
    {
      fprintf(stderr, "Packets to receive = %d\n", count);
    }
  conn->packet_count = count;
  conn->packet_count_total = count;
  conn->xfer_done = 0;
}

/* Process transfer completed packet */
void tgps_xfer_cmplt(Tgps conn, TgpsPacket packet)
{
  size_t size;
  int command_id;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_INT, &command_id,
		     TGPS_FORMAT_END);
  if (size != packet->data_len)
    fprintf(stderr, "Warning: extra junk after packet\n");
  if (conn->packet_count == 0)
    {
      if (isatty(fileno(stderr)))
	fprintf(stderr, " %d/%d \n",
		conn->packet_count_total - conn->packet_count,
		conn->packet_count_total);
      if (conn->verbose)
	fprintf(stderr, "All packets received for type %d\n", command_id);
    }
  else if (conn->packet_count < 0)
    fprintf(stderr,
	    "Warning: Too many packets received for type %d, %d "
	    "extra packets\n",
	    command_id, -conn->packet_count);
  else 
    fprintf(stderr, "Warning: Too few packets received for "
	    "type %d, %d missing packets\n",
	    command_id, conn->packet_count);
  if (!(conn->flags & TGPS_FLAGS_DO_NOT_EXIT))
    conn->xfer_done = 1;
  conn->packet_count = 0;
  conn->packet_count_total = 0;
}

/* Process packet */
void tgps_process_packet(Tgps conn, TgpsPacket packet)
{
  TgpsWaypointStruct w;
  TgpsRouteHeaderStruct r;
  TgpsTrackStruct t;
  TgpsRouteLnkStruct rl;
  TgpsTrackHeaderStruct th;
  TgpsPositionStruct pos;
  TgpsAlmanacStruct a;
  TgpsTimeStruct tim;
  TgpsPvtStruct pvt;
  int ack = 0, ret = 0;

  switch (packet->data[0])
    {
    case TGPS_PID_L000_ACK_BYTE:
      tgps_ack_byte(conn, packet);
      break;
    case TGPS_PID_L000_NAK_BYTE:
      tgps_nak_byte(conn, packet);
      break;
    case TGPS_PID_L000_PROTOCOL_ARRAY:
      ack = 1;
      tgps_protocol_array(conn, packet);
      break;
    case TGPS_PID_L000_PRODUCT_RQST:
      ack = 1;
      /* XXX, Ignored now, should not be sent by the GPS device */
      break;
    case TGPS_PID_L000_PRODUCT_DATA:
      tgps_product_data_in(conn, packet);
      ack = 1;
      break;
    case TGPS_PID_L001_COMMAND_DATA:
    case TGPS_PID_L002_COMMAND_DATA:
      ack = 1;
      /* XXX, Ignored now, no responder support */
      break;
    case TGPS_PID_L001_XFER_CMPLT: /* case TGPS_PID_L002_XFER_CMPLT: */
      ack = 1;
      tgps_xfer_cmplt(conn, packet);
      break;
    case TGPS_PID_L001_RECORDS: /* case TGPS_PID_L002_RECORDS: */
      ack = 1;
      tgps_records(conn, packet);
      break;
    case TGPS_PID_L001_WPT_DATA:
    case TGPS_PID_L002_WPT_DATA:
    case TGPS_PID_L001_RTE_WPT_DATA:
    case TGPS_PID_L002_RTE_WPT_DATA:
      ack = 1;
      ret = tgps_wpt_data_in(conn, packet, &w);
      if (ret)
	{
	  tgps_print_waypoint(conn, &w);
	  tgps_free_waypoint(&w);
	}
      else
	{
	  if (conn->verbose > 1)
	    fprintf(stderr, "Warning: Could not decode waypoint packet\n");
	}
      break;
    case TGPS_PID_L001_RTE_HDR:
    case TGPS_PID_L002_RTE_HDR:
      ack = 1;
      ret = tgps_rte_hdr_data_in(conn, packet, &r);
      if (ret)
	{
	  tgps_print_route_header(conn, &r);
	  tgps_free_route_header(&r);
	}
      else
	{
	  if (conn->verbose > 1)
	    fprintf(stderr, "Warning: Could not decode route header packet\n");
	}
      break;
    case TGPS_PID_L001_RTE_LINK_DATA:
      ack = 1;
      ret = tgps_rte_lnk_data_in(conn, packet, &rl);
      if (ret)
	{
	  tgps_print_route_link(conn, &rl);
	  tgps_free_route_link(&rl);
	}
      else
	{
	  if (conn->verbose > 1)
	    fprintf(stderr, "Warning: Could not decode route link packet\n");
	}
      break;
    case TGPS_PID_L001_PRX_WPT_DATA:
      ack = 1;
      ret = tgps_prx_wpt_data_in(conn, packet, &w);
      if (ret)
	{
	  tgps_print_waypoint(conn, &w);
	  tgps_free_waypoint(&w);
	}
      else
	{
	  if (conn->verbose > 1)
	    fprintf(stderr, "Warning: Could not decode proximity waypoint packet\n");
	}
      break;
    case TGPS_PID_L001_TRK_DATA:
      ack = 1;
      ret = tgps_track_data_in(conn, packet, &t);
      if (ret)
	{
	  tgps_print_track(conn, &t);
	}
      else
	{
	  if (conn->verbose > 1)
	    fprintf(stderr, "Warning: Could not decode track packet\n");
	}
      break;
    case TGPS_PID_L001_TRK_HDR:
      ack = 1;
      ret = tgps_track_hdr_data_in(conn, packet, &th);
      if (ret)
	{
	  tgps_print_track_header(conn, &th);
	  tgps_free_track_header(&th);
	}
      else
	{
	  if (conn->verbose > 1)
	    fprintf(stderr, "Warning: Could not decode track header packet\n");
	}
      break;
    case TGPS_PID_L001_ALMANAC_DATA:
    case TGPS_PID_L002_ALMANAC_DATA:
      ack = 1;
      ret = tgps_almanac_data_in(conn, packet, &a);
      if (ret)
	{
	  tgps_print_almanac(conn, &a);
	  tgps_free_almanac(&a);
	}
      else
	{
	  if (conn->verbose > 1)
	    fprintf(stderr, "Warning: Could not decode almanac packet\n");
	}
      break;
    case TGPS_PID_L001_DATE_TIME_DATA:
    case TGPS_PID_L002_DATE_TIME_DATA:
      ack = 1;
      ret = tgps_time_data_in(conn, packet, &tim);
      if (ret)
	{
	  tgps_print_time(conn, &tim);
	}
      else
	{
	  if (conn->verbose > 1)
	    fprintf(stderr, "Warning: Could not decode time packet\n");
	}
      break;
    case TGPS_PID_L001_POSITION_DATA:
    case TGPS_PID_L002_POSITION_DATA:
      ack = 1;
      ret = tgps_position_data_in(conn, packet, &pos);
      if (ret)
	{
	  tgps_print_position(conn, &pos);
	}
      else
	{
	  if (conn->verbose > 1)
	    fprintf(stderr, "Warning: Could not decode position packet\n");
	}
      break;
    case TGPS_PID_L001_PVT_DATA:
      ret = tgps_pvt_data_in(conn, packet, &pvt);
      if (ret)
	{
	  tgps_print_pvt(conn, &pvt);
	}
      else
	{
	  if (conn->verbose > 1)
	    fprintf(stderr, "Warning: Could not decode pvt packet\n");
	}
      break;
    }
  if (ack)
    tgps_send_ack(conn, packet->data[0]);
}

/* If this is set then abort the program */
int tgps_do_abort = 0;

/* Signal handler that will abort the transfer */
void tgps_abort_transfer(int sig)
{
  struct sigaction sa;
  
  sa.sa_handler = SIG_DFL;
  sigemptyset(&(sa.sa_mask));
  sa.sa_flags = SA_RESTART;
  
  tgps_do_abort = 1;
  sigaction(SIGINT, &sa, NULL);
}

/* Send abort operation to gps. */
void tgps_abort(Tgps conn)
{
  tgps_do_abort = 0;
  if (conn->pvt_active)
    {
      tgps_encode_send_packet(conn, TGPS_PID_COMMAND_DATA(conn),
			      TGPS_FORMAT_INT,
			      TGPS_CMND_STOP_PVT_DATA(conn), 
			      TGPS_FORMAT_END);
      conn->pvt_active = 0;
    }
  if (conn->upload & 0xff)
    {
      tgps_encode_send_packet(conn, TGPS_PID_XFER_CMPL(conn),
			      TGPS_FORMAT_INT,
			      (conn->upload & 0xff),
			      TGPS_FORMAT_END);
    }
  if (conn->operation != TGPS_UPLOAD &&
      conn->operation != TGPS_START_PVT &&
      conn->operation != TGPS_OPERATION_NONE)
    {      
      tgps_encode_send_packet(conn, TGPS_PID_COMMAND_DATA(conn),
			      TGPS_FORMAT_INT,
			      TGPS_CMND_ABORT_TRANSFER(conn), 
			      TGPS_FORMAT_END);
    }
  conn->xfer_done = 1;
  conn->upload = 0;
}

/* Start transfer */
void tgps_start_transfer(Tgps conn, int command)
{
  if ((conn->upload & 0xff) != 0)
    {
      tgps_encode_send_packet(conn, TGPS_PID_XFER_CMPL(conn),
			      TGPS_FORMAT_INT,
			      (conn->upload & 0xff),
			      TGPS_FORMAT_END);
    }
  conn->upload &= ~0xff;
  conn->upload |= command & 0xff;
}

/* Get line */
TgpsPacket tgps_get_line(Tgps conn)
{
  TgpsWaypointStruct w;
  TgpsRouteHeaderStruct r;
  TgpsTrackStruct t;
  TgpsRouteLnkStruct rl;
  TgpsTrackHeaderStruct th;
  TgpsPositionStruct pos;
  TgpsAlmanacStruct a;
  TgpsTimeStruct tim;
  TgpsPacket p = NULL;

  do {
    conn->input_buffer_len = -1;
    tgps_clear_waypoint(&w);
    tgps_clear_route_header(&r);
    tgps_clear_route_link(&rl);
    tgps_clear_track(&t);
    tgps_clear_track_header(&th);
    if (tgps_get_waypoint(conn, &w))
      {
	if ((conn->upload & 0xff) != TGPS_CMND_TRANSFER_WPT(conn) &&
	    (conn->upload & 0xff) != TGPS_CMND_TRANSFER_RTE(conn))
	  {
	    if (w.proximity > TGPS_INVALID_FLOAT_TEST && w.proximity != 0.0 &&
		w.proximity < 0.9e+25)
	      tgps_start_transfer(conn, TGPS_CMND_TRANSFER_PRX(conn));
	    else
	      tgps_start_transfer(conn, TGPS_CMND_TRANSFER_WPT(conn));
	  }
				   
	tgps_defaults_waypoint(&w);
	p = tgps_wpt_data_out(conn, &w);
      }
    else if (tgps_get_route_header(conn, &r))
      {
	if ((conn->upload & 0xff) != TGPS_CMND_TRANSFER_RTE(conn))
	  {
	    tgps_start_transfer(conn, TGPS_CMND_TRANSFER_RTE(conn));
	  }

	tgps_defaults_route_header(&r);
	p = tgps_rte_hdr_data_out(conn, &r);
      }
    else if (tgps_get_track(conn, &t))
      {
	if ((conn->upload & 0xff) != TGPS_CMND_TRANSFER_TRK(conn))
	  {
	    tgps_start_transfer(conn, TGPS_CMND_TRANSFER_TRK(conn));
	  }
	tgps_defaults_track(&t);
	p = tgps_track_data_out(conn, &t);
      }
    else if (tgps_get_route_link(conn, &rl))
      {
	if ((conn->upload & 0xff) != TGPS_CMND_TRANSFER_RTE(conn))
	  {
	    tgps_start_transfer(conn, TGPS_CMND_TRANSFER_RTE(conn));
	  }
	tgps_defaults_route_link(&rl);
	p = tgps_rte_lnk_data_out(conn, &rl);
      }
    else if (tgps_get_track_hdr(conn, &th))
      {
	if ((conn->upload & 0xff) != TGPS_CMND_TRANSFER_TRK(conn))
	  {
	    tgps_start_transfer(conn, TGPS_CMND_TRANSFER_TRK(conn));
	  }
	tgps_defaults_track_header(&th);
	p = tgps_track_hdr_data_out(conn, &th);
      }
    else if (tgps_get_clock(conn, &tim))
      {
	tgps_start_transfer(conn, 0);
	p = tgps_time_data_out(conn, &tim);
      }
    else if (tgps_get_position(conn, &pos))
      {
	tgps_start_transfer(conn, 0);
	p = tgps_position_data_out(conn, &pos);
      }
    else if (tgps_get_almanac(conn, &a))
      {
	if ((conn->upload & 0xff) != TGPS_CMND_TRANSFER_ALM(conn))
	  {
	    tgps_start_transfer(conn, TGPS_CMND_TRANSFER_ALM(conn));
	  }
	p = tgps_almanac_data_out(conn, &a);
      }
    tgps_free_track_header(&th);
    tgps_free_route_link(&rl);
    tgps_free_waypoint(&w);
    tgps_free_route_header(&r);
    if (p)
      break;
  } while (conn->input_buffer_len != 0);
  return p;
}

/* Write data to output file */
void tgps_print(Tgps conn, const char *format, ...)
{
  va_list ap;

  va_start(ap, format);
  vfprintf(conn->output_stream, format, ap);
  va_end(ap);
}

/* Get line from input file, and put it to input buffer. Returns true if
   successfull, and false if end of file. */
int tgps_gets(Tgps conn)
{
  return (fgets(conn->input_buffer, conn->input_buffer_alloc,
		conn->input_stream) == NULL);
}

/* Main program */
int main(int argc, char **argv)
{
  Tgps conn;
  char *tty = getenv("GPSDEV");
  char *output = "-";
  char *input = "-";
  int ret, product_id = 0;
  int cmd_sent = 0;
  int c, errflg = 0, flags = 0, verbose = 0, version = 0;
  TgpsOperation operation = TGPS_OPERATION_NONE;
  TgpsDataVersion data_version = TGPS_DATA_VERSION_TGPS_V1;
  TgpsDegreeFormat degree_format = TGPS_DEG_NONE;
  struct sigaction sa;
  time_t t, prev = 0 ;

  if (tty == NULL)
    {
      tty = "/dev/gps0";
    }

  while ((c = getopt(argc, argv, "gdmsuwrtacCpPlDf:o:i:IvVhEF:")) != EOF)
    {
      switch (c)
	{
	case 'g':
	  data_version = TGPS_DATA_VERSION_GPSTRANS;
	  break;
	case 'd':
	  if (degree_format != TGPS_DEG_NONE)
	    {
	      fprintf(stderr,
		      "Only one of -d, -m, or -s options may be given\n");
	      errflg++;
	    }
	  degree_format = TGPS_DEG;
	  break;
	case 'm':
	  if (degree_format != TGPS_DEG_NONE)
	    {
	      fprintf(stderr,
		      "Only one of -d, -m, or -s options may be given\n");
	      errflg++;
	    }
	  degree_format = TGPS_DEG_MIN;
	  break;
	case 's':
	  if (degree_format != TGPS_DEG_NONE)
	    {
	      fprintf(stderr,
		      "Only one of -d, -m, or -s options may be given\n");
	      errflg++;
	    }
	  degree_format = TGPS_DEG_MIN_SEC;
	  break;
	case 'u':
	  if (operation != TGPS_OPERATION_NONE)
	    {
	      fprintf(stderr, "Only one command code may be given\n");
	      errflg++;
	    }
	  operation = TGPS_UPLOAD;
	  break;
	case 'w':
	  if (operation != TGPS_OPERATION_NONE)
	    {
	      fprintf(stderr, "Only one command code may be given\n");
	      errflg++;
	    }
	  operation = TGPS_DOWNLOAD_WAYPOINTS;
	  break;
	case 'P':
	  if (operation != TGPS_OPERATION_NONE)
	    {
	      fprintf(stderr, "Only one command code may be given\n");
	      errflg++;
	    }
	  operation = TGPS_DOWNLOAD_PROXIMITY;
	  break;
	case 'r':
	  if (operation != TGPS_OPERATION_NONE)
	    {
	      fprintf(stderr, "Only one command code may be given\n");
	      errflg++;
	    }
	  operation = TGPS_DOWNLOAD_ROUTES;
	  break;
	case 't':
	  if (operation != TGPS_OPERATION_NONE)
	    {
	      fprintf(stderr, "Only one command code may be given\n");
	      errflg++;
	    }
	  operation = TGPS_DOWNLOAD_TRACK;
	  break;
	case 'a':
	  if (operation != TGPS_OPERATION_NONE)
	    {
	      fprintf(stderr, "Only one command code may be given\n");
	      errflg++;
	    }
	  operation = TGPS_DOWNLOAD_ALMANAC;
	  break;
	case 'c':
	  if (operation != TGPS_OPERATION_NONE)
	    {
	      fprintf(stderr, "Only one command code may be given\n");
	      errflg++;
	    }
	  operation = TGPS_DOWNLOAD_TIME;
	  break;
	case 'C':
	  if (operation != TGPS_OPERATION_NONE)
	    {
	      fprintf(stderr, "Only one command code may be given\n");
	      errflg++;
	    }
	  operation = TGPS_SET_TIME;
	  break;
	case 'p':
	  if (operation != TGPS_OPERATION_NONE)
	    {
	      fprintf(stderr, "Only one command code may be given\n");
	      errflg++;
	    }
	  operation = TGPS_DOWNLOAD_POSITION;
	  break;
	case 'l':
	  if (operation != TGPS_OPERATION_NONE)
	    {
	      fprintf(stderr, "Only one command code may be given\n");
	      errflg++;
	    }
	  operation = TGPS_START_PVT;
	  break;
	case 'D':
	  tgps_packet_dump = 1;
	  break;
	case 'f':
	  tty = optarg;
	  break;
	case 'o':
	  output = optarg;
	  break;
	case 'i':
	  input = optarg;
	  break;
	case 'I':
	  flags |= TGPS_FLAGS_IGNORE_COMPRESSED_TRACKS;
	  break;
	case 'E':
	  flags |= TGPS_FLAGS_IGNORE_ERRORS;
	  break;
	case 'v':
	  verbose++;
	  break;
	case 'F':
	  product_id = atoi(optarg);
	  break;
	case 'V':
	  version = 1;
	  break;
	case 'h':
	case '?':
	  errflg = 1;
	  break;
	}
    }

  if (degree_format == TGPS_DEG_NONE)
    degree_format = TGPS_DEG;

  if (version)
    {
      if (verbose == 0)
	fprintf(stderr, "Tgps version %s\n", VERSION);
      else if (verbose == 1)
	fprintf(stderr, "Tgps version %s by Tero Kivinen\n", VERSION);
      else if (verbose == 2)
	fprintf(stderr, "Tgps version %s by Tero Kivinen <kivinen@iki.fi>\n",
		VERSION);
      else if (verbose == 3)
	fprintf(stderr, "Tgps version %s by "
		"Tero T. Kivinen <kivinen@iki.fi>\n", VERSION);
      else
	{
	  fprintf(stderr, "Tgps version %s by Tero T. Kivinen "
		  "<kivinen@iki.fi>, http://www.iki.fi/kivinen/\n", VERSION);
	  if (verbose > 10)
	    fprintf(stderr, "No, I will not give you more "
		    "information about myself.\n");
	}
      exit(0);
    }
  if (errflg || argc - optind != 0)
    {
      fprintf(stderr, "Usage: tgps [-V] [-v] [-D] [-I] [-g] [-d | -m | -s] "
	      "[-w | -r | -t | -a | -c | -p | -u | -l] "
	      "[-f gps_dev] [-o output_file] [-i input_file]\n");
      exit(1);
    }
  conn = tgps_open(tty, output, input);

  if (conn == NULL)
    {
      fprintf(stderr, "tgps_open failed\n");
      exit(1);
    }

  if (product_id != 0)
    conn->product_id = product_id;

  if (isatty(fileno(conn->output_stream)) && operation != TGPS_SET_TIME)
    {
      prev = -1;
      verbose = 0;
    }

  conn->operation = operation;
  conn->degree_format = degree_format;
  conn->data_version = data_version;
  conn->flags = flags;
  conn->verbose = verbose;

  sa.sa_handler = tgps_abort_transfer;
  sigemptyset(&(sa.sa_mask));
  sa.sa_flags = SA_RESTART;
  sigaction(SIGINT, &sa, NULL);
  
  tgps_print_header(conn);

  tgps_send_packet(conn, TGPS_PID_L000_PRODUCT_RQST, NULL, 0);

  while (1)
    {
      int write_blocked, read_blocked;

      ret = tgps_wait(conn, 1, 0);
      if (ret < 0)
	goto error;
      if (ret == 0)
	{
	  int i, cnt;

	  /* Resend first packet for each different command in output queue */
	  cnt = 0;
	  for(i = 0; i < 256; i++)
	    {
	      TgpsPacket packet;

	      packet = conn->command[i];
	      if (!packet)
		continue;
	      if (conn->verbose > 3)
		fprintf(stderr, "Warning: Resending packet of type %d, "
			"retry = %d/%d\n", i, packet->retry_cnt,
			TGPS_RETRY_LIMIT);
	      if (packet->retry_cnt++ >= TGPS_RETRY_LIMIT)
		{
		  if (conn->verbose > 1)
		    fprintf(stderr, "Warning: Resend retry limit exceeded.\n");
#ifdef LINUX_SERIAL_FIX
		  if (conn->verbose > 1)
		    fprintf(stderr, "Warning: Reopening the tty device\n");
		  close(conn->tty);
		  if (tgps_serial_open(conn, tty) == 0)
		    {
		      close(conn->tty);
		      fprintf(stderr, "Warning: Cannot reopen tty\n");
		      exit(1);
		    }
#else  /* LINUX_SERIAL_FIX */
		  if (conn->verbose > 1)
		    fprintf(stderr, "Warning: Dropping the packet\n");
		  
		  conn->command[i] = packet->next;
		  tgps_free_packet(conn, packet);
		  continue;
#endif /* LINUX_SERIAL_FIX */
		}
	      tgps_add_packet(conn, packet);
	      cnt++;
	    }
	  if (cnt == 0 && cmd_sent == 0)
	    {
	      conn->xfer_done = 0;
	      switch (conn->operation)
		{
		case TGPS_UPLOAD:
		  tgps_encode_send_packet(conn, TGPS_PID_RECORDS(conn),
					  TGPS_FORMAT_INT, 0,
					  TGPS_FORMAT_END);
		  conn->upload = TGPS_UPLOAD_ACTIVATED;
		  break;
		case TGPS_DOWNLOAD_WAYPOINTS:
		  tgps_encode_send_packet(conn, TGPS_PID_COMMAND_DATA(conn),
					  TGPS_FORMAT_INT,
					  TGPS_CMND_TRANSFER_WPT(conn), 
					  TGPS_FORMAT_END);
		  break;
		case TGPS_DOWNLOAD_PROXIMITY:
		  tgps_encode_send_packet(conn, TGPS_PID_COMMAND_DATA(conn),
					  TGPS_FORMAT_INT,
					  TGPS_CMND_TRANSFER_PRX(conn), 
					  TGPS_FORMAT_END);
		  break;
		case TGPS_DOWNLOAD_ROUTES:
		  tgps_encode_send_packet(conn, TGPS_PID_COMMAND_DATA(conn),
					  TGPS_FORMAT_INT,
					  TGPS_CMND_TRANSFER_RTE(conn), 
					  TGPS_FORMAT_END);
		  break;
		case TGPS_DOWNLOAD_TRACK:
		  tgps_encode_send_packet(conn, TGPS_PID_COMMAND_DATA(conn),
					  TGPS_FORMAT_INT,
					  TGPS_CMND_TRANSFER_TRK(conn), 
					  TGPS_FORMAT_END);
		  break;
		case TGPS_DOWNLOAD_ALMANAC:
		  tgps_encode_send_packet(conn, TGPS_PID_COMMAND_DATA(conn),
					  TGPS_FORMAT_INT,
					  TGPS_CMND_TRANSFER_ALM(conn), 
					  TGPS_FORMAT_END);
		  break;
		case TGPS_DOWNLOAD_TIME:
		case TGPS_SET_TIME:
		  tgps_encode_send_packet(conn, TGPS_PID_COMMAND_DATA(conn),
					  TGPS_FORMAT_INT,
					  TGPS_CMND_TRANSFER_TIME(conn), 
					  TGPS_FORMAT_END);
		  conn->packet_count = 1;
		  break;
		case TGPS_DOWNLOAD_POSITION:
		  tgps_encode_send_packet(conn, TGPS_PID_COMMAND_DATA(conn),
					  TGPS_FORMAT_INT,
					  TGPS_CMND_TRANSFER_POSN(conn), 
					  TGPS_FORMAT_END);
		  conn->packet_count = 1;
		  break;
		case TGPS_START_PVT:
		  tgps_encode_send_packet(conn, TGPS_PID_COMMAND_DATA(conn),
					  TGPS_FORMAT_INT,
					  TGPS_CMND_START_PVT_DATA(conn), 
					  TGPS_FORMAT_END);
		  conn->pvt_active = 1;
		  break;
		case TGPS_OPERATION_NONE:
		  conn->flags |= TGPS_FLAGS_DO_NOT_EXIT;
		  break;
		}
	      cmd_sent = 1;
	    }
	  else if (cnt == 0 && conn->pvt_active)
	    {
	      conn->pvt_active++;
	      if (conn->pvt_active > TGPS_PVT_RETRY)
		{
		  char timebuf[30];
		  time_t t;
#ifdef HAVE_STRFTIME
		  struct tm *tptr;

		  t = time(NULL);
		  tptr = localtime(&t);
		  strftime(timebuf, sizeof(timebuf), "%F %T", tptr);
#else /* HAVE_STRFTIME */
		  t = time(NULL);
		  strcpy(timebuf, ctime(&t));
#endif /* HAVE_STRFTIME */
		  fprintf(stderr, "PVT active, but nothing received for "
			  "60 seconds. Trying to restart %s\n",
			  timebuf);
		  conn->pvt_active = 0;
		  tgps_send_packet(conn, TGPS_PID_L000_PRODUCT_RQST, NULL, 0);
		  cmd_sent = 0;
		}
	    }
	  else if (cnt == 0 && cmd_sent == 1 && conn->xfer_done == 1)
	    break;
	}

      write_blocked = 0;
      read_blocked = 0;
      while (!read_blocked)
	{
	  TgpsPacket packet;

	  if (tgps_do_abort)
	    tgps_abort(conn);

	  ret = tgps_read(conn);
	  if (ret == 0)
	    read_blocked = 1;
	  else if (ret < 0)
	    goto error;
	  else
	    write_blocked = 0;
	  while ((packet = tgps_find_packet(conn)))
	    {
	      if (tgps_do_abort)
		tgps_abort(conn);

#if 0
	      tgps_print_buffer("Processing: ",
				packet->data, packet->data_len);
#endif
	      tgps_process_packet(conn, packet);
	      tgps_free_packet(conn, packet);

	      t = time(NULL);
	      if (prev != -1 && t - prev > 1)
		{
		  if (conn->packet_count > 0 &&
		      isatty(fileno(stderr)))
		    fprintf(stderr, " %d/%d \r",
			    conn->packet_count_total - conn->packet_count,
			    conn->packet_count_total);
		  prev = t;
		}

	      if (!write_blocked)
		{
		  if (conn->upload &&
		      (conn->upload & TGPS_UPLOAD_WAITING_FOR_ACK) == 0)
		    {
		      TgpsPacket p;
		      p = tgps_get_line(conn);
		      if (p == NULL)
			{
			  tgps_do_abort = 1;
			}
		      else
			{
			  if (isatty(fileno(stderr)))
			    fprintf(stderr, ".");
			  tgps_send_complete_packet(conn, p);
			  conn->upload |= TGPS_UPLOAD_WAITING_FOR_ACK;
			  tgps_free_packet(conn, p);
			}
		    }
		  ret = tgps_write(conn);
		  if (ret == 0)
		    write_blocked = 1;
		  else if (ret < 0)
		    goto error;
		  else
		    read_blocked = 0;
		}
	    }

	  /* Clear the duplicate detection buffer */
	  conn->buffer_last_packet_len = 0;

	  if (tgps_do_abort)
	    tgps_abort(conn);

	  if (!write_blocked)
	    {
	      if (conn->upload &&
		  (conn->upload & TGPS_UPLOAD_WAITING_FOR_ACK) == 0)
		{
		  TgpsPacket p;
		  p = tgps_get_line(conn);
		  if (p == NULL)
		    {
		      tgps_do_abort = 1;
		    }
		  else
		    {
		      if (isatty(fileno(stderr)))
			fprintf(stderr, ".");
		      tgps_send_complete_packet(conn, p);
		      conn->upload |= TGPS_UPLOAD_WAITING_FOR_ACK;
		      tgps_free_packet(conn, p);
		    }
		}
	      ret = tgps_write(conn);
	      if (ret == 0)
		write_blocked = 1;
	      else if (ret < 0)
		goto error;
	      else
		read_blocked = 0;
	    }
	}
    }
 error:
  if (tgps_close(conn) < 0)
    {
      fprintf(stderr, "tgps_close failed\n");
      exit(1);
    }
  return 0;
}
