/*
 * tracks.c -- GPS protocol track handling.
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/tracks.c,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 21:36 Apr 24 2000 kivinen
 *	  Last Modification : 18:29 Feb  5 2004 kivinen
 *	  Last check in     : $Date: 2003/08/18 20:08:54 $
 *	  Revision number   : $Revision: 1.9 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.140
 *	  Edit time	    : 45 min
 *
 *	  Description       : GPS protocol track handling
 *
 *	  $Log: tracks.c,v $
 *	  Revision 1.9  2003/08/18 20:08:54  kivinen
 *	  	Added support for GPS V. Fixed packet size and actual size
 *	  	comparisions.
 *
 *	  Revision 1.8  2001/08/23 15:43:46  kivinen
 *	  	Added support for eTrex format using format number 309 (my own
 *	  	number).
 *
 *	  Revision 1.7  2000/08/05 00:43:43  kivinen
 *	  	Updated to new interface.
 *
 *	  Revision 1.6  2000/07/21 22:16:21  kivinen
 *	  	Moved stuff to data.c.
 *
 *	  Revision 1.5  2000/07/12 22:19:53  kivinen
 *	  	Final upload support.
 *
 *	  Revision 1.4  2000/07/06 23:07:42  kivinen
 *	  	Added initial upload support.
 *
 *	  Revision 1.3  2000/04/30 01:35:29  kivinen
 *	  	Added missing else.
 *
 *	  Revision 1.2  2000/04/30 01:03:36  kivinen
 *	  	Updated to rev 03 document.
 *
 *	  Revision 1.1  2000/04/29 16:40:57  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#include "tgps.h"
#include "packet.h"
#include "data.h"

int tgps_track_data_in_d300(Tgps conn, TgpsPacket packet, TgpsTrack t);
int tgps_track_data_in_d301(Tgps conn, TgpsPacket packet, TgpsTrack t);
int tgps_track_data_in_d309(Tgps conn, TgpsPacket packet, TgpsTrack t);
TgpsPacket tgps_track_data_out_d300(Tgps conn, TgpsTrack t);
TgpsPacket tgps_track_data_out_d301(Tgps conn, TgpsTrack t);
TgpsPacket tgps_track_data_out_d309(Tgps conn, TgpsTrack t);
int tgps_track_hdr_data_in_d310(Tgps conn, TgpsPacket packet,
				TgpsTrackHeader t);
TgpsPacket tgps_track_hdr_data_out_d310(Tgps conn, TgpsTrackHeader t);

/* Process version 300 track log type */
int tgps_track_data_in_d300(Tgps conn, TgpsPacket packet, TgpsTrack t)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_SEMICIRCLE, &t->lat, &t->lon,
		     TGPS_FORMAT_LONGWORD, &t->time,
		     TGPS_FORMAT_BOOLEAN, &t->new_track,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 301 track log type */
int tgps_track_data_in_d301(Tgps conn, TgpsPacket packet, TgpsTrack t)
{
  size_t size;

  if (packet->data_len == 26)
    return tgps_track_data_in_d309(conn, packet, t);
  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_SEMICIRCLE, &t->lat, &t->lon,
		     TGPS_FORMAT_LONGWORD, &t->time,
		     TGPS_FORMAT_FLOAT, &t->alt,
		     TGPS_FORMAT_FLOAT, &t->dpth,
		     TGPS_FORMAT_BOOLEAN, &t->new_track,
		     TGPS_FORMAT_END);
  if (t->alt >= 1.0e20)
    t->alt = TGPS_INVALID_FLOAT;
  if (t->dpth >= 1.0e20)
    t->dpth = TGPS_INVALID_FLOAT;

  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 309 track log type XXX check this */
int tgps_track_data_in_d309(Tgps conn, TgpsPacket packet, TgpsTrack t)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_SEMICIRCLE, &t->lat, &t->lon,
		     TGPS_FORMAT_LONGWORD, &t->time,
		     TGPS_FORMAT_FLOAT, &t->alt,
		     TGPS_FORMAT_FLOAT, &t->dpth,
		     TGPS_FORMAT_BOOLEAN, &t->new_track,
		     TGPS_FORMAT_BYTE_ARRAY, t->extra, 3,
		     TGPS_FORMAT_END);
  if (t->alt >= 1.0e20)
    t->alt = TGPS_INVALID_FLOAT;
  if (t->dpth >= 1.0e20)
    t->dpth = TGPS_INVALID_FLOAT;

  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process track point packet */
int tgps_track_data_in(Tgps conn, TgpsPacket packet, TgpsTrack t)
{
  int ret;

  tgps_clear_track(t);

  if (tgps_is_supported(conn, 'D', 300) || packet->data_len == 15)
    ret = tgps_track_data_in_d300(conn, packet, t);
  else if (tgps_is_supported(conn, 'D', 301) || packet->data_len == 23)
    ret = tgps_track_data_in_d301(conn, packet, t);
  else if (tgps_is_supported(conn, 'D', 309) || packet->data_len == 26)
    ret = tgps_track_data_in_d309(conn, packet, t);
  else
    {
      fprintf(stderr, "Unknown track point data\n");
      return 0;
    }
  if (ret != 0)
    conn->packet_count--;
  return ret;
}

/* Send version 300 track log data packet */
TgpsPacket tgps_track_data_out_d300(Tgps conn, TgpsTrack t)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_TRK_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_SEMICIRCLE, t->lat, t->lon,
		     TGPS_FORMAT_LONGWORD, t->time,
		     TGPS_FORMAT_BOOLEAN, t->new_track,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 301 track log data packet */
TgpsPacket tgps_track_data_out_d301(Tgps conn, TgpsTrack t)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_TRK_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_SEMICIRCLE, t->lat, t->lon,
		     TGPS_FORMAT_LONGWORD, t->time,
		     TGPS_FORMAT_FLOAT, t->alt,
		     TGPS_FORMAT_FLOAT, t->dpth,
		     TGPS_FORMAT_BOOLEAN, t->new_track,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 309 track log data packet XXX */
TgpsPacket tgps_track_data_out_d309(Tgps conn, TgpsTrack t)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_TRK_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_SEMICIRCLE, t->lat, t->lon,
		     TGPS_FORMAT_LONGWORD, t->time,
		     TGPS_FORMAT_FLOAT, t->alt,
		     TGPS_FORMAT_FLOAT, t->dpth,
		     TGPS_FORMAT_BOOLEAN, t->new_track,
		     TGPS_FORMAT_BYTE_ARRAY, t->extra, 3,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send track log packet */
TgpsPacket tgps_track_data_out(Tgps conn, TgpsTrack t)
{
  if (tgps_is_supported(conn, 'D', 300))
    return tgps_track_data_out_d300(conn, t);
  else if (tgps_is_supported(conn, 'D', 301))
    return tgps_track_data_out_d301(conn, t);
  else if (tgps_is_supported(conn, 'D', 309))
    return tgps_track_data_out_d309(conn, t);
  else
    {
      fprintf(stderr, "Unknown track log type\n");
      return NULL;
    }
}

/* Process version 310 track header type */
int tgps_track_hdr_data_in_d310(Tgps conn, TgpsPacket packet,
				TgpsTrackHeader t)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_BOOLEAN, &t->display,
		     TGPS_FORMAT_BYTE, &t->color,
		     TGPS_FORMAT_STRING, &t->ident,
		     TGPS_FORMAT_END);
  if (t->color <= 0)
    t->color = 0;
  else
    t->color += 16;

  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process track header packet */
int tgps_track_hdr_data_in(Tgps conn, TgpsPacket packet, TgpsTrackHeader t)
{
  int ret;

  tgps_clear_track_header(t);

  if (tgps_is_supported(conn, 'D', 310))
    ret = tgps_track_hdr_data_in_d310(conn, packet, t);
  else
    {
      fprintf(stderr, "Unknown track header type\n");
      return 0;
    }
  if (ret != 0)
    conn->packet_count--;
  return ret;
}

/* Send version 310 track header data packet */
TgpsPacket tgps_track_hdr_data_out_d310(Tgps conn, TgpsTrackHeader t)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_TRK_HDR(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_BOOLEAN, t->display,
		     TGPS_FORMAT_BYTE, t->color,
		     TGPS_FORMAT_STRING, t->ident,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send track log packet */
TgpsPacket tgps_track_hdr_data_out(Tgps conn, TgpsTrackHeader t)
{
  if (tgps_is_supported(conn, 'D', 310))
    return tgps_track_hdr_data_out_d310(conn, t);
  else
    {
      fprintf(stderr, "Unknown track log type\n");
      return NULL;
    }
}
