/*
 * util.c -- Utility functions
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/util.c,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 09:24 Jul 21 2000 kivinen
 *	  Last Modification : 12:34 Aug 19 2003 kivinen
 *	  Last check in     : $Date: 2003/08/18 20:09:04 $
 *	  Revision number   : $Revision: 1.3 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.16
 *	  Edit time	    : 8 min
 *
 *	  Description       : Utility functions
 *
 *	  $Log: util.c,v $
 *	  Revision 1.3  2003/08/18 20:09:04  kivinen
 *	  	Added support for GPS V. Fixed packet size and actual size
 *	  	comparisions.
 *
 *	  Revision 1.2  2000/08/05 00:44:54  kivinen
 *	  	Removed warnings. Moved stuff from tgps to here.
 *
 *	  Revision 1.1  2000/07/21 22:15:55  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#include "tgps.h"
#include "ieee-double.h"

/* Convert semicircle to degrees */
double tgps_semicircle_to_deg(long semicircle)
{
  return (double) semicircle * (180.0 / 2147483648.0);
}

/* Convert degrees to semicircle */
long tgps_deg_to_semicircle(double degrees)
{
  return (long) (degrees * (2147483648.0 / 180.0));
}

/* Convert radians to degrees */
double tgps_radians_to_deg(double radians)
{
  return (double) radians * (180.0 / M_PI);
}

/* Convert degrees to radians */
double tgps_deg_to_radians(double degrees)
{
  return (double) degrees * (M_PI / 180.0);
}

/* Is character array empty */
int tgps_is_empty(char *str)
{
  while (*str == ' ')
    str++;
  if (*str)
    return 0;
  return 1;
}

/* Print buffer. */
void tgps_print_buffer(char *txt, unsigned char *buffer, size_t buflen)
{
  size_t i, j;

  if (txt)
    fprintf(stderr, "%s\n", txt);
  for(i = 0; i < buflen; i += 16)
    {
      fprintf(stderr, "%08x: ", i);
      for(j = 0; j < 16; j++)
	{
	  if (i + j < buflen)
	    fprintf(stderr, "%02x", buffer[i + j]);
	  else
	    fprintf(stderr, "  ");
	  if ((j % 2) == 1)
	    fprintf(stderr, " ");
	}
      fprintf(stderr, "  ");
      for(j = 0; j < 16; j++)
	{
	  if (i + j < buflen)
	    if (isprint(buffer[i + j]))
	      fprintf(stderr, "%c", buffer[i + j]);
	    else
	      fprintf(stderr, ".");
	  else
	    fprintf(stderr, " ");
	}
      fprintf(stderr, "\n");
    }
}

/* Put packet back to free list */
void tgps_free_packet(Tgps conn, TgpsPacket packet)
{
  packet->next = conn->free_list;
  conn->free_list = packet;
}

/* Get packet from the free list */
TgpsPacket tgps_get_packet(Tgps conn, size_t size)
{
  TgpsPacket packet;

  packet = conn->free_list;
  if (packet == NULL)
    {
      packet = calloc(1, sizeof(*packet));
      if (packet == NULL)
	return NULL;
    }
  else
    {
      conn->free_list = packet->next;
      packet->next = NULL;
    }

 retry:
  if (packet->data == NULL)
    {
      packet->alloc_size = (size | 0x7) + 1;
      packet->data = malloc(packet->alloc_size);
    }
  else if (packet->alloc_size < size)
    {
      packet->alloc_size = (size | 0x7) + 1;
      packet->data = realloc(packet->data, packet->alloc_size);
    }

  if (packet->data == NULL)
    {
      free(packet);
      packet = conn->free_list;
      if (packet == NULL)
	return NULL;
      conn->free_list = packet->next;
      packet->next = NULL;
      goto retry;
    }
  packet->data_len = 0;
  packet->retry_cnt = 0;
  return packet;
}

/* Encode data to the buffer using varargs list. Return number of bytes written
   to buffer, or -1 if the data didn't fit to the buffer */
size_t tgps_encode_va(unsigned char *buffer, size_t buffer_len, va_list ap)
{
  TgpsFormat format;
  unsigned char *p, buf[8], *size_ptr = NULL;
  unsigned long l;
  double d;
  float f;
  size_t len, i, len_avail;

#define ADD_BYTE(buf,len,ch) \
  do { *(buf)++ = (ch); if (--(len) == 0) goto error; } while (0)

  len_avail = buffer_len;
  while (len_avail > 0)
    {
      format = va_arg(ap, TgpsFormat);
      switch (format)
	{
	case TGPS_FORMAT_LEN:
	  size_ptr = buffer;
	  ADD_BYTE(buffer, len_avail, 0);
	  break;
	case TGPS_FORMAT_CHAR:
	  ADD_BYTE(buffer, len_avail, va_arg(ap, int));
	  break;
	case TGPS_FORMAT_BYTE:
	  ADD_BYTE(buffer, len_avail, va_arg(ap, unsigned int));
	  break;
	case TGPS_FORMAT_INT:
	  l = va_arg(ap, int);
	  ADD_BYTE(buffer, len_avail, l & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 8) & 0xff);
	  break;
	case TGPS_FORMAT_WORD:
	  l = va_arg(ap, unsigned int);
	  ADD_BYTE(buffer, len_avail, l & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 8) & 0xff);
	  break;
	case TGPS_FORMAT_LONG:
	  l = va_arg(ap, long);
	  ADD_BYTE(buffer, len_avail, l & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 8) & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 16) & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 24) & 0xff);
	  break;
	case TGPS_FORMAT_LONGWORD:
	  l = va_arg(ap, unsigned long);
	  ADD_BYTE(buffer, len_avail, l & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 8) & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 16) & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 24) & 0xff);
	  break;
	case TGPS_FORMAT_FLOAT:
	  f = va_arg(ap, double);
	  tgps_f2buf_lsb(buf, f);
	  ADD_BYTE(buffer, len_avail, buf[0]);
	  ADD_BYTE(buffer, len_avail, buf[1]);
	  ADD_BYTE(buffer, len_avail, buf[2]);
	  ADD_BYTE(buffer, len_avail, buf[3]);
	  break;
	case TGPS_FORMAT_DOUBLE:
	  d = va_arg(ap, double);
	  tgps_d2buf_lsb(buf, d);
	  ADD_BYTE(buffer, len_avail, buf[0]);
	  ADD_BYTE(buffer, len_avail, buf[1]);
	  ADD_BYTE(buffer, len_avail, buf[2]);
	  ADD_BYTE(buffer, len_avail, buf[3]);
	  ADD_BYTE(buffer, len_avail, buf[4]);
	  ADD_BYTE(buffer, len_avail, buf[5]);
	  ADD_BYTE(buffer, len_avail, buf[6]);
	  ADD_BYTE(buffer, len_avail, buf[7]);
	  break;
	case TGPS_FORMAT_CHAR_ARRAY:
	  p = va_arg(ap, unsigned char *);
	  len = va_arg(ap, size_t);

	  for(i = 0; *p && i < len; i++)
	    {
	      ADD_BYTE(buffer, len_avail, *p++);
	    }
	  for(; i < len; i++)
	    {
	      ADD_BYTE(buffer, len_avail, ' ');
	    }
	  break;
	case TGPS_FORMAT_BYTE_ARRAY:
	  p = va_arg(ap, unsigned char *);
	  len = va_arg(ap, size_t);

	  for(i = 0; i < len; i++)
	    {
	      ADD_BYTE(buffer, len_avail, *p++);
	    }
	  break;
	case TGPS_FORMAT_STRING:
	  p = va_arg(ap, unsigned char *);

	  while (p && *p)
	    {
	      ADD_BYTE(buffer, len_avail, *p++);
	    }
	  ADD_BYTE(buffer, len_avail, 0);
	  break;
	case TGPS_FORMAT_BOOLEAN:
	  ADD_BYTE(buffer, len_avail, va_arg(ap, int));
	  break;
	case TGPS_FORMAT_SEMICIRCLE:
	  l = va_arg(ap, long);
	  ADD_BYTE(buffer, len_avail, l & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 8) & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 16) & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 24) & 0xff);
	  l = va_arg(ap, long);
	  ADD_BYTE(buffer, len_avail, l & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 8) & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 16) & 0xff);
	  ADD_BYTE(buffer, len_avail, (l >> 24) & 0xff);
	  break;
	case TGPS_FORMAT_RADIAN:
	  d = va_arg(ap, double);
	  tgps_d2buf_lsb(buf, d);
	  ADD_BYTE(buffer, len_avail, buf[0]);
	  ADD_BYTE(buffer, len_avail, buf[1]);
	  ADD_BYTE(buffer, len_avail, buf[2]);
	  ADD_BYTE(buffer, len_avail, buf[3]);
	  ADD_BYTE(buffer, len_avail, buf[4]);
	  ADD_BYTE(buffer, len_avail, buf[5]);
	  ADD_BYTE(buffer, len_avail, buf[6]);
	  ADD_BYTE(buffer, len_avail, buf[7]);

	  d = va_arg(ap, double);
	  tgps_d2buf_lsb(buf, d);
	  ADD_BYTE(buffer, len_avail, buf[0]);
	  ADD_BYTE(buffer, len_avail, buf[1]);
	  ADD_BYTE(buffer, len_avail, buf[2]);
	  ADD_BYTE(buffer, len_avail, buf[3]);
	  ADD_BYTE(buffer, len_avail, buf[4]);
	  ADD_BYTE(buffer, len_avail, buf[5]);
	  ADD_BYTE(buffer, len_avail, buf[6]);
	  ADD_BYTE(buffer, len_avail, buf[7]);
	  break;
	case TGPS_FORMAT_END:
	  goto out;
	}
    }
#undef ADD_BYTE
 out:
  if (size_ptr)
    *size_ptr = buffer_len - len_avail;
  *buffer = '\0';
  return buffer_len - len_avail;
 error:
  return -1;
}

/* Encode data to the buffer using varargs list. Return number of bytes written
   to buffer, or -1 if the data didn't fit to the buffer */
size_t tgps_encode(unsigned char *buffer, size_t buffer_len, ...)
{
  size_t bytes;
  va_list ap;

  va_start(ap, buffer_len);
  bytes = tgps_encode_va(buffer, buffer_len, ap);
  va_end(ap);
  return bytes;
}

/* Decode data from the buffer using varargs list. Return number of bytes
   read from buffer, or -1 if there was not enough data in buffer */
size_t tgps_decode_va(unsigned char *buffer, size_t buffer_len, va_list ap)
{
  TgpsFormat format;
  unsigned char *p, **pp;
  unsigned int *ip;
  unsigned long l, *lp;
  double *dp;
  float *fp;
  size_t i, len_avail;

#define CHECK_LEN(len, size) do { if ((len) < (size)) goto error; } while (0)
#define GET_BYTE(buf, len) ((len)--, *(buf)++)

  len_avail = buffer_len;
  while (len_avail > 0)
    {
      format = va_arg(ap, TgpsFormat);
      switch (format)
	{
	case TGPS_FORMAT_CHAR:
	  p = va_arg(ap, char *);
	  CHECK_LEN(len_avail, 1);
	  if (p == NULL)
	    (void) GET_BYTE(buffer, len_avail);
	  else
	    *p = GET_BYTE(buffer, len_avail);
	  break;
	case TGPS_FORMAT_BYTE:
	  p = va_arg(ap, unsigned char *);
	  CHECK_LEN(len_avail, 1);
	  if (p == NULL)
	    (void) GET_BYTE(buffer, len_avail);
	  else
	    *p = GET_BYTE(buffer, len_avail);
	  break;
	case TGPS_FORMAT_INT:
	  ip = va_arg(ap, int *);
	  CHECK_LEN(len_avail, 2);
	  l = GET_BYTE(buffer, len_avail);
	  l |= GET_BYTE(buffer, len_avail) << 8;
	  if (ip != NULL)
	    *ip = (unsigned int) l;
	  break;
	case TGPS_FORMAT_WORD:
	  ip = va_arg(ap, unsigned int *);
	  CHECK_LEN(len_avail, 2);
	  l = GET_BYTE(buffer, len_avail);
	  l |= GET_BYTE(buffer, len_avail) << 8;
	  if (ip != NULL)
	    *ip = (unsigned int) l;
	  break;
	case TGPS_FORMAT_LONG:
	  lp = va_arg(ap, long *);
	  CHECK_LEN(len_avail, 4);
	  l = GET_BYTE(buffer, len_avail);
	  l |= GET_BYTE(buffer, len_avail) << 8;
	  l |= GET_BYTE(buffer, len_avail) << 16;
	  l |= GET_BYTE(buffer, len_avail) << 24;
	  if (lp != NULL)
	    *lp = (long) l;
	  break;
	case TGPS_FORMAT_LONGWORD:
	  lp = va_arg(ap, unsigned long *);
	  CHECK_LEN(len_avail, 4);
	  l = GET_BYTE(buffer, len_avail);
	  l |= GET_BYTE(buffer, len_avail) << 8;
	  l |= GET_BYTE(buffer, len_avail) << 16;
	  l |= GET_BYTE(buffer, len_avail) << 24;
	  if (lp != NULL)
	    *lp = (unsigned long) l;
	  break;
	case TGPS_FORMAT_FLOAT:
	  fp = va_arg(ap, float *);
	  CHECK_LEN(len_avail, 4);
	  if (fp != NULL)
	    *fp = tgps_buf2f_lsb(buffer);
	  len_avail -= 4;
	  buffer += 4;
	  break;
	case TGPS_FORMAT_DOUBLE:
	  dp = va_arg(ap, double *);
	  CHECK_LEN(len_avail, 8);
	  if (dp != NULL)
	    *dp = tgps_buf2d_lsb(buffer);
	  len_avail -= 8;
	  buffer += 8;
	  break;
	case TGPS_FORMAT_CHAR_ARRAY:
	case TGPS_FORMAT_BYTE_ARRAY:
	  p = va_arg(ap, unsigned char *);
	  i = va_arg(ap, size_t);

	  CHECK_LEN(len_avail, i);
	  while (i-- > 0)
	    {
	      if (p == NULL)
		(void) GET_BYTE(buffer, len_avail);
	      else
		*p++ = GET_BYTE(buffer, len_avail);
	    }
	  break;
	case TGPS_FORMAT_STRING:
	  pp = va_arg(ap, unsigned char **);

	  for(i = 0; buffer[i] && i < len_avail; i++)
	    ;
	  if (i < len_avail)
	    i++;
	  if (pp != NULL)
	    {
	      *pp = p = malloc(i);
	      if (p == NULL)
		return -1;
	      while (--i > 0)
		*p++ = GET_BYTE(buffer, len_avail);
	      *p++ = GET_BYTE(buffer, len_avail);
	    }
	  else
	    {
	      while (--i > 0)
		(void) GET_BYTE(buffer, len_avail);
	      *p++ = GET_BYTE(buffer, len_avail);
	    }
	  break;
	case TGPS_FORMAT_BOOLEAN:
	  ip = va_arg(ap, int *);
	  CHECK_LEN(len_avail, 1);
	  if (ip == NULL)
	    {
	      (void) GET_BYTE(buffer, len_avail);
	    }
	  else
	    {
	      if (GET_BYTE(buffer, len_avail))
		*ip = 1;
	      else
		*ip = 0;
	    }
	  break;
	case TGPS_FORMAT_SEMICIRCLE:
	  lp = va_arg(ap, long *);
	  CHECK_LEN(len_avail, 4);
	  l = GET_BYTE(buffer, len_avail);
	  l |= GET_BYTE(buffer, len_avail) << 8;
	  l |= GET_BYTE(buffer, len_avail) << 16;
	  l |= GET_BYTE(buffer, len_avail) << 24;
	  if (lp != NULL)
	    *lp = (long) l;
	  lp = va_arg(ap, long *);
	  CHECK_LEN(len_avail, 4);
	  l = GET_BYTE(buffer, len_avail);
	  l |= GET_BYTE(buffer, len_avail) << 8;
	  l |= GET_BYTE(buffer, len_avail) << 16;
	  l |= GET_BYTE(buffer, len_avail) << 24;
	  if (lp != NULL)
	    *lp = (long) l;
	  break;
	case TGPS_FORMAT_RADIAN:
	  dp = va_arg(ap, double *);
	  CHECK_LEN(len_avail, 8);
	  if (dp != NULL)
	    *dp = tgps_buf2d_lsb(buffer);
	  len_avail -= 8;
	  buffer += 8;

	  dp = va_arg(ap, double *);
	  CHECK_LEN(len_avail, 8);
	  if (dp != NULL)
	    *dp = tgps_buf2d_lsb(buffer);
	  len_avail -= 8;
	  buffer += 8;
	  break;
	case TGPS_FORMAT_LEN:
	  fprintf(stderr, "Internal error: TGPS_FORMAT_LEN found while decoding\n");
	  exit(1);
	  break;
	case TGPS_FORMAT_END:
	  goto out;
	}
    }
#undef ADD_BYTE
 out:
  return buffer_len - len_avail;
 error:
  return -1;
}

/* Decode data from the buffer using varargs list. Return number of bytes
   read from buffer, or -1 if there was not enough data in buffer */
size_t tgps_decode(unsigned char *buffer, size_t buffer_len, ...)
{
  size_t bytes;
  va_list ap;

  va_start(ap, buffer_len);
  bytes = tgps_decode_va(buffer, buffer_len, ap);
  va_end(ap);
  return bytes;
}

